/* conf.c */

/* configuration adjustment. Some of the ideas and bits and pieces of the
 * code here are based on TinyMUSH 2.0.
 */

#include "copyrite.h"
#include "config.h"

#include <stdio.h>
#ifdef I_SYS_TIME
#include <sys/time.h>
#else
#include <time.h>
#endif
#ifdef I_STRING
#include <string.h>
#else
#include <strings.h>
#endif
#ifdef I_STDLIB
#include <stdlib.h>
#endif
#include <ctype.h>

#include "externs.h"
#include "conf.h"
#include "intrface.h"
#include "mushdb.h"
#include "confmagic.h"

time_t mudtime;			/* game time, in seconds */

extern object_flag_type find_flag _((char *name, int type, int *toggle, int is_conf));
extern int restrict_command _((char *name, char *restriction));

OPTTAB options;

typedef struct confparm CONF;

struct confparm {
  char *name;			/* name of option */
  /* the function handler */
  void (*handler) ();
/* This should be:
 * void (*handler) _((char *opt, char *val, int *loc, int maxval));
 * but some compilers (ultrix 4.2, for example) totally barf, *sigh*
 */
  int *loc;			/* place to put this option */
  int max;			/* max: string length, integer value */
};


void cf_bool _((char *opt, char *val, int *loc, int maxval));
void cf_str _((char *opt, char *val, int *loc, int maxlen));
void cf_int _((char *opt, char *val, int *loc, int maxval));
void cf_flag _((char *opt, char *val, int *loc, int maxval));
void config_set _((char *opt, char *val));
void conf_default_set _((void));
int config_file_startup _((const char *conf));

CONF conftable[] =
{
  {(char *) "mud_name", cf_str, (int *) options.mud_name, 128},
  {(char *) "port", cf_int, &options.port, 32000},
  {(char *) "input_database", cf_str, (int *) options.input_db, 256},
  {(char *) "output_database", cf_str, (int *) options.output_db, 256},
  {(char *) "crash_database", cf_str, (int *) options.crash_db, 256},
  {(char *) "mail_database", cf_str, (int *) options.mail_db, 256},
  {(char *) "player_start", cf_int, &options.player_start, 100000},
  {(char *) "master_room", cf_int, &options.master_room, 100000},
  {(char *) "base_room", cf_int, &options.base_room, 1000000},
  {(char *) "idle_timeout", cf_int, &options.idle_timeout, 100000},
  {(char *) "dump_interval", cf_int, &options.dump_interval, 100000},
  {(char *) "dump_message", cf_str, (int *) options.dump_message, 256},
  {(char *) "dump_complete", cf_str, (int *) options.dump_complete, 256},
  {(char *) "ident_timeout", cf_int, &options.ident_timeout, 60},
  {(char *) "max_logins", cf_int, &options.max_logins, 128},
  {(char *) "whisper_loudness", cf_int, &options.whisper_loudness, 100},
  {(char *) "paycheck", cf_int, &options.paycheck, 1000},
  {(char *) "starting_money", cf_int, &options.starting_money, 10000},
  {(char *) "starting_quota", cf_int, &options.starting_quota, 10000},
  {(char *) "player_queue_limit", cf_int, &options.player_queue_limit, 100000},
  {(char *) "queue_chunk", cf_int, &options.queue_chunk, 100000},
{(char *) "active_queue_chunk", cf_int, &options.active_q_chunk, 100000},
  {(char *) "function_recursion_limit", cf_int, &options.func_nest_lim, 100000},
  {(char *) "function_invocation_limit", cf_int, &options.func_invk_lim, 100000},
  {(char *) "money_singular", cf_str, (int *) options.money_singular, 32},
  {(char *) "money_plural", cf_str, (int *) options.money_plural, 32},
  {(char *) "dbcomp", cf_bool, &options.dbcomp, 2},
{(char *) "compress_program", cf_str, (int *) options.compressprog, 256},
  {(char *) "uncompress_program", cf_str, (int *) options.uncompressprog, 256},
  {(char *) "help_file", cf_str, (int *) options.help_file, 256},
  {(char *) "help_index", cf_str, (int *) options.help_index, 256},
  {(char *) "news_file", cf_str, (int *) options.news_file, 256},
  {(char *) "news_index", cf_str, (int *) options.news_index, 256},
  {(char *) "events_file", cf_str, (int *) options.events_file, 256},
  {(char *) "events_index", cf_str, (int *) options.events_index, 256},
#if (CHAT_SYSTEM > 3)
  {(char *) "chat_database", cf_str, (int *) options.chatdb, 256},
  {(char *) "max_player_chans", cf_int, &options.max_player_chans, 100},
  {(char *) "max_channels", cf_int, &options.max_channels, 1000},
  {(char *) "chan_cost", cf_int, &options.chan_cost, 10000},
#endif
  {(char *) "connect_file", cf_str, (int *) options.connect_file, 256},
  {(char *) "motd_file", cf_str, (int *) options.motd_file, 256},
  {(char *) "wizmotd_file", cf_str, (int *) options.wizmotd_file, 256},
  {(char *) "newuser_file", cf_str, (int *) options.newuser_file, 256},
  {(char *) "register_create_file", cf_str, (int *) options.register_file, 256},
  {(char *) "quit_file", cf_str, (int *) options.quit_file, 256},
  {(char *) "down_file", cf_str, (int *) options.down_file, 256},
  {(char *) "full_file", cf_str, (int *) options.full_file, 256},
  {(char *) "guest_file", cf_str, (int *) options.guest_file, 256},
  {(char *) "log_commands", cf_bool, &options.log_commands, 2},
  {(char *) "log_huhs", cf_bool, &options.log_huhs, 2},
  {(char *) "log_forces", cf_bool, &options.log_forces, 2},
  {(char *) "log_walls", cf_bool, &options.log_walls, 2},
  {(char *) "logins", cf_bool, &options.login_allow, 2},
  {(char *) "player_creation", cf_bool, &options.create_allow, 2},
  {(char *) "guests", cf_bool, &options.guest_allow, 2},
  {(char *) "daytime", cf_bool, &options.daytime, 2},
  {(char *) "player_flags", cf_flag, &options.player_flags, 64},
  {(char *) "room_flags", cf_flag, &options.room_flags, 64},
  {(char *) "exit_flags", cf_flag, &options.exit_flags, 64},
  {(char *) "thing_flags", cf_flag, &options.thing_flags, 64},
  {(char *) "rwho_dump_interval", cf_int, &options.rwho_interval, 32000},
  {(char *) "rwho_info_port", cf_int, &options.rwho_port, 32000},
  {(char *) "rwho_host", cf_str, (int *) options.rwho_host, 64},
  {(char *) "rwho_password", cf_str, (int *) options.rwho_pass, 64},
#ifdef USE_WARNINGS
  {(char *) "warn_interval", cf_int, &options.warn_interval, 32000},
#endif
  {(char *) "use_dns", cf_bool, &options.use_dns, 2},
  {NULL, NULL, NULL, 0}
};

void
cf_bool(opt, val, loc, maxval)
    char *opt;
    char *val;
    int *loc;
    int maxval;
{
  /* enter boolean parameter */

  if (!strcasecmp(val, "yes") || !strcasecmp(val, "true") ||
      !strcasecmp(val, "1"))
    *loc = 1;
  else if (!strcasecmp(val, "no") || !strcasecmp(val, "false") ||
	   !strcasecmp(val, "0"))
    *loc = 0;
  else {
    fprintf(stderr, "CONFIGURATION: option %s value %s invalid.\n", opt, val);
    fflush(stderr);
  }
}


void
cf_str(opt, val, loc, maxlen)
    char *opt;
    char *val;
    int *loc;
    int maxlen;
{
  /* enter string parameter */

  /* truncate if necessary */
  if (strlen(val) >= (Size_t) maxlen) {
    val[maxlen - 1] = '\0';
    fprintf(stderr, "CONFIGURATION: option %s value truncated\n", opt);
    fflush(stderr);
  }
  strcpy((char *) loc, val);
}


void
cf_int(opt, val, loc, maxval)
    char *opt;
    char *val;
    int *loc;
    int maxval;
{
  /* enter integer parameter */

  int n;

  n = atoi(val);

  /* enforce limits */
  if (n > maxval) {
    n = maxval;
    fprintf(stderr, "CONFIGURATION: option %s value limited to %d\n",
	    opt, maxval);
    fflush(stderr);
  }
  *loc = n;
}

void
cf_flag(opt, val, loc, maxval)
    char *opt;
    char *val;
    int *loc;
    int maxval;
{
  /* set default flags */

  int f = -1;
  int toggle;

  /* figure out what flag type we're setting */

  switch (opt[0]) {
  case 'p':
    f = find_flag(val, TYPE_PLAYER, &toggle, 1);
    break;
  case 'r':
    f = find_flag(val, TYPE_ROOM, &toggle, 1);
    break;
  case 'e':
    f = find_flag(val, TYPE_EXIT, &toggle, 1);
    break;
  case 't':
    f = find_flag(val, TYPE_THING, &toggle, 1);
    break;
  default:
    fprintf(stderr, "CONFIGURATION: weird flag set directive '%s'\n", opt);
  }

  if (f == -1) {
    fprintf(stderr, "CONFIGURATION: flag '%s' cannot be set.\n", val);
    return;
  }
  if (f == -2) {
    fprintf(stderr, "CONFIGURATION: flag '%s' for type not found.\n", val);
    return;
  }
  if (!toggle)
    *loc |= f;
  else {
    switch (opt[0]) {
    case 'p':
      options.player_toggles |= f;
      break;
    case 'r':
      options.room_toggles |= f;
      break;
    case 'e':
      options.exit_toggles |= f;
      break;
    case 't':
      options.thing_toggles |= f;
      break;
    }
  }
}

void
config_set(opt, val)
    char *opt;
    char *val;
{
  CONF *cp;
  char *p;

  /* Was this "restrict_command <command> <restriction>"? If so, do it */
  if (!strcasecmp(opt, "restrict_command")) {
    for (p = val; *p && !isspace(*p); p++) ;
    if (*p) {
      *p++ = '\0';
      if (!restrict_command(val, p)) {
	fprintf(stderr, "CONFIGURATION: Invalid command or restriction for %s.\n", val);
	fflush(stderr);
      }
    } else {
      fprintf(stderr, "CONFIGURATION: restrict_command %s requires a restriction value.\n", val);
      fflush(stderr);
    }
    return;
  }
  /* search conf table for the option; if found, add it, if not found,
   * complain about it.
   */
  for (cp = conftable; cp->name; cp++) {
    if (!strcmp(cp->name, opt)) {
      cp->handler(opt, val, cp->loc, cp->max);
      return;
    }
  }

  fprintf(stderr, "CONFIGURATION: directive '%s' in cnf file ignored.\n", opt);
  fflush(stderr);
}

void
conf_default_set()
{
  strcpy(options.mud_name, "TinyMUSH");
  options.port = 4201;
  strcpy(options.input_db, "data/indb.Z");
  strcpy(options.output_db, "data/outdb.Z");
  strcpy(options.crash_db, "data/PANIC.db");
#if (CHAT_SYSTEM > 3)
  strcpy(options.chatdb, "data/chatdb.Z");
  options.chan_cost = 1000;
  options.max_player_chans = 3;
  options.max_channels = 200;
#endif
  strcpy(options.mail_db, "data/maildb.Z");
  options.player_start = 0;
  options.master_room = 2;
  options.idle_timeout = 10801;
  options.dump_interval = 3601;
  strcpy(options.dump_message, "GAME: Dumping database. Game may freeze for a minute");
  strcpy(options.dump_complete, "GAME: Dump complete. Time in.");
  options.ident_timeout = 5;
  options.max_logins = 128;
  options.whisper_loudness = 100;
  options.paycheck = 50;
  options.starting_money = 100;
  options.starting_quota = 20;
  options.player_queue_limit = 100;
  options.queue_chunk = 3;
  options.active_q_chunk = 0;
  options.func_nest_lim = 50;
  options.func_invk_lim = 2500;
  strcpy(options.money_singular, "Penny");
  strcpy(options.money_plural, "Pennies");
  options.dbcomp = 1;
  strcpy(options.compressprog, "compress");
  strcpy(options.uncompressprog, "uncompress");
  strcpy(options.help_file, "txt/help.txt");
  strcpy(options.help_index, "txt/help.idx");
  strcpy(options.news_file, "txt/news.txt");
  strcpy(options.news_index, "txt/news.idx");
  strcpy(options.events_file, "txt/events.txt");
  strcpy(options.events_index, "txt/events.idx");
  strcpy(options.connect_file, "txt/connect.txt");
  strcpy(options.motd_file, "txt/motd.txt");
  strcpy(options.wizmotd_file, "txt/wizmotd.txt");
  strcpy(options.newuser_file, "txt/newuser.txt");
  strcpy(options.register_file, "txt/register.txt");
  strcpy(options.quit_file, "txt/quit.txt");
  strcpy(options.down_file, "txt/down.txt");
  strcpy(options.full_file, "txt/full.txt");
  strcpy(options.guest_file, "txt/guest.txt");
  options.log_commands = 0;
  options.log_huhs = 0;
  options.log_forces = 1;
  options.log_walls = 0;
  options.login_allow = 1;
  options.guest_allow = 1;
  options.create_allow = 1;
  options.daytime = 0;
  options.player_flags = 0;
  options.room_flags = 0;
  options.exit_flags = 0;
  options.thing_flags = 0;
  options.player_toggles = 0;
  options.room_toggles = 0;
  options.exit_toggles = 0;
  options.thing_toggles = 0;
  options.rwho_interval = 241;
  options.rwho_port = 6889;
  strcpy(options.rwho_host, "littlewood.math.okstate.edu");
  strcpy(options.rwho_pass, "getyours");
#ifdef USE_WARNINGS
  options.warn_interval = 3600;
#endif
  options.use_dns = 1;
}

int
config_file_startup(conf)
    const char *conf;
{
  /* read a configuration file. Return 0 on failure, 1 on success */

  FILE *fp;
  char tbuf1[BUFFER_LEN];
  char *p, *q, *s;

  static char cfile[BUFFER_LEN];	/* Remember the last one */
  if (conf && *conf)
    strcpy(cfile, conf);
  fp = fopen(cfile, "r");

  if (fp == NULL) {
    do_rawlog(LT_ERR, "ERROR: Cannot open configuration file %s.\n",
	      cfile);
    return 0;
  }
  conf_default_set();		/* initialize defaults */

  fgets(tbuf1, BUFFER_LEN, fp);
  while (!feof(fp)) {

    p = tbuf1;

    if (*p == '#') {
      /* comment line */
      fgets(tbuf1, BUFFER_LEN, fp);
      continue;
    }
    /* this is a real line. Strip the newline and characters following it.
     * Split the line into command and argument portions. If it exists,
     * also strip off the trailing comment.
     */

    for (p = tbuf1; *p && (*p != '\n'); p++) ;
    *p = '\0';			/* strip '\n' */
    for (p = tbuf1; *p && isspace(*p); p++)	/* strip spaces */
      ;
    for (q = p; *q && !isspace(*q); q++)	/* move over command */
      ;
    if (*q)
      *q++ = '\0';		/* split off command */
    for (; *q && isspace(*q); q++)	/* skip spaces */
      ;
    for (s = q; *s && (*s != '#'); s++)		/* look for comment */
      ;
    if (*s)			/* if found nuke it */
      *s = '\0';
    for (s = s - 1; (s >= q) && isspace(*s); s--)	/* smash trailing stuff */
      *s = '\0';

    if (strlen(p) != 0)		/* skip blank lines */
      config_set(p, q);

    fgets(tbuf1, BUFFER_LEN, fp);
  }

  /* these directives aren't player-settable but need to be initialized */
  mudtime = time(NULL);
  options.dump_counter = mudtime + options.dump_interval;
#ifdef RWHO_SEND
  options.rwho_counter = mudtime + options.rwho_interval;
#endif
#ifdef USE_WARNINGS
  options.warn_counter = mudtime + options.warn_interval;
#endif

  fclose(fp);
  return 1;
}
