/* htab.c - table hashing routines */

/* This code is ripped out of TinyMUSH 2.2. */
/* Minor tweaking to make in Penn-compatible by Trivian (xeno@mix.hive.no) */


#include "config.h"
#include "copyrite.h"
#include "externs.h"
#include "intrface.h"
#include "htab.h"
#include "confmagic.h"


/* ---------------------------------------------------------------------------
 * hashval: Compute hash value of a string for a hash table.
 */

int
hashval(str, hashmask)
    const char *str;
    int hashmask;
{
  int hash = 0;
  const char *sp;

  /*
   * If the string pointer is null, return 0.  If not, add up the
   * numeric value of all the characters and return the sum,
   * modulo the size of the hash table.
   */

  if (str == NULL)
    return 0;
  for (sp = str; *sp; sp++)
    hash = (hash << 5) + hash + *sp;
  return (hash & hashmask);
}

/* ----------------------------------------------------------------------
 * get_hashmask: Get hash mask for mask-style hashing.
 */

int
get_hashmask(size)
    int *size;
{
  int tsize;

  /* Get next power-of-two >= size, return power-1 as the mask
   * for ANDing
   */

  for (tsize = 1; tsize < *size; tsize = tsize << 1) ;
  *size = tsize;
  return tsize - 1;
}

/* ---------------------------------------------------------------------------
 * hashinit: Initialize a new hash table.
 */

void
hashinit(htab, size)
    HASHTAB *htab;
    int size;
{
  int i;

  htab->mask = get_hashmask(&size);
  htab->hashsize = size;
  htab->checks = 0;
  htab->scans = 0;
  htab->max_scan = 0;
  htab->hits = 0;
  htab->entries = 0;
  htab->deletes = 0;
  htab->nulls = size;
  htab->entry = (HASHARR *) mush_malloc(size * sizeof(struct hashentry *),
					"hashinit");

  for (i = 0; i < size; i++)
    htab->entry->element[i] = NULL;
}

/* ---------------------------------------------------------------------------
 * hashreset: Reset hash table stats.
 */

void
hashreset(htab)
    HASHTAB *htab;
{
  htab->checks = 0;
  htab->scans = 0;
  htab->hits = 0;
}

/* ---------------------------------------------------------------------------
 * hashfind: Look up an entry in a hash table and return a pointer to its
 * hash data.
 */

void *
hashfind(str, htab)
    const char *str;
    HASHTAB *htab;
{
  int hval, numchecks;
  HASHENT *hptr;

  numchecks = 0;
  htab->scans++;
  hval = hashval(str, htab->mask);
  for (hptr = htab->entry->element[hval]; hptr != NULL; hptr = hptr->next) {
    numchecks++;
    if (strcmp(str, hptr->target) == 0) {
      htab->hits++;
      if (numchecks > htab->max_scan)
	htab->max_scan = numchecks;
      htab->checks += numchecks;
      return hptr->data;
    }
  }
  if (numchecks > htab->max_scan)
    htab->max_scan = numchecks;
  htab->checks += numchecks;
  return (void *) NULL;
}

/* ---------------------------------------------------------------------------
 * hashadd: Add a new entry to a hash table.
 */

int
hashadd(str, hashdata, htab)
    const char *str;
    void *hashdata;
    HASHTAB *htab;
{
  int hval;
  HASHENT *hptr;

  /*
   * Make sure that the entry isn't already in the hash table.  If it
   * is, exit with an error.  Otherwise, create a new hash block and
   * link it in at the head of its thread.
   */

  if (hashfind(str, htab) != NULL)
    return -1;
  hval = hashval(str, htab->mask);
  htab->entries++;
  if (htab->entry->element[hval] == NULL)
    htab->nulls--;
  hptr = (HASHENT *) mush_malloc(sizeof(HASHENT), "hashent");
  hptr->target = (char *) strdup(str);
  hptr->data = hashdata;
  hptr->next = htab->entry->element[hval];
  htab->entry->element[hval] = hptr;
  return 0;
}

/* ---------------------------------------------------------------------------
 * hashdelete: Remove an entry from a hash table.
 */

void
hashdelete(str, htab)
    char *str;
    HASHTAB *htab;
{
  int hval;
  HASHENT *hptr, *last;

  hval = hashval(str, htab->mask);
  last = NULL;
  for (hptr = htab->entry->element[hval];
       hptr != NULL;
       last = hptr, hptr = hptr->next) {
    if (strcmp(str, hptr->target) == 0) {
      if (last == NULL)
	htab->entry->element[hval] = hptr->next;
      else
	last->next = hptr->next;
      free(hptr->target);
      mush_free(hptr, "hashent");
      htab->deletes++;
      htab->entries--;
      if (htab->entry->element[hval] == NULL)
	htab->nulls++;
      return;
    }
  }
}

/* ---------------------------------------------------------------------------
 * hashflush: free all the entries in a hashtable.
 */

void
hashflush(htab, size)
    HASHTAB *htab;
    int size;
{
  HASHENT *hent, *thent;
  int i;

  for (i = 0; i < htab->hashsize; i++) {
    hent = htab->entry->element[i];
    while (hent != NULL) {
      thent = hent;
      hent = hent->next;
      free(thent->target);
      mush_free(thent, "hashent");
    }
    htab->entry->element[i] = NULL;
  }

  /* Resize if needed.  Otherwise, just zero all the stats */

  if ((size > 0) && (size != htab->hashsize)) {
    mush_free(htab->entry, "hashinit");
    hashinit(htab, size);
  } else {
    htab->checks = 0;
    htab->scans = 0;
    htab->max_scan = 0;
    htab->hits = 0;
    htab->entries = 0;
    htab->deletes = 0;
    htab->nulls = htab->hashsize;
  }
}

/* --------------------------------------------------------------------------
 * hashfree: flush a hash table and free its memory. Totally destroys
 *           the table -- you must hashinit after this if you want it back
 */
void
hashfree(htab)
    HASHTAB *htab;
{
  hashflush(htab, 0);
  mush_free(htab->entry, "hashinit");
}

/* ---------------------------------------------------------------------------
 * hashrepl: replace the data part of a hash entry.
 */

int
hashrepl(str, hashdata, htab)
    char *str;
    void *hashdata;
    HASHTAB *htab;
{
  HASHENT *hptr;
  int hval;

  hval = hashval(str, htab->mask);
  for (hptr = htab->entry->element[hval];
       hptr != NULL;
       hptr = hptr->next) {
    if (strcmp(str, hptr->target) == 0) {
      hptr->data = hashdata;
      return 0;
    }
  }
  return -1;
}

/* ---------------------------------------------------------------------------
 * hashinfo: return an mbuf with hashing stats
 */

char *
hashinfo(tab_name, htab)
    const char *tab_name;
    HASHTAB *htab;
{
  char *buff;

  buff = mush_malloc(BUFFER_LEN, "hashinfo");
  sprintf(buff, "%-15s %4d%8d%8d%8d%8d%8d%8d%8d",
	  tab_name, htab->hashsize, htab->entries, htab->deletes,
	  htab->nulls, htab->scans, htab->hits, htab->checks,
	  htab->max_scan);
  return buff;
}

/* Returns the key for the first hash entry in 'htab'. */

void *
hash_firstentry(htab)
    HASHTAB *htab;
{
  int hval;

  for (hval = 0; hval < htab->hashsize; hval++)
    if (htab->entry->element[hval] != NULL) {
      htab->last_hval = hval;
      htab->last_entry = htab->entry->element[hval];
      return htab->entry->element[hval]->data;
    }
  return (void *) NULL;
}

void *
hash_nextentry(htab)
    HASHTAB *htab;
{
  int hval;
  HASHENT *hptr;

  hval = htab->last_hval;
  hptr = htab->last_entry;
  if (hptr->next != NULL) {	/* We can stay in the same chain */
    htab->last_entry = hptr->next;
    return hptr->next->data;
  }
  /* We were at the end of the previous chain, go to the next one */
  hval++;
  while (hval < htab->hashsize) {
    if (htab->entry->element[hval] != NULL) {
      htab->last_hval = hval;
      htab->last_entry = htab->entry->element[hval];
      return htab->entry->element[hval]->data;
    }
    hval++;
  }
  return (void *) NULL;
}

/* ---------------------------------------------------------------------------
 * nhashfind: Look up an entry in a numeric hash table and return a pointer
 * to its hash data.
 */

void *
nhashfind(val, htab)
    int val;
    NHSHTAB *htab;
{
  int hval, numchecks;
  NHSHENT *hptr;

  numchecks = 0;
  htab->scans++;
  hval = (val & htab->mask);
  for (hptr = htab->entry->element[hval]; hptr != NULL; hptr = hptr->next) {
    numchecks++;
    if (val == hptr->target) {
      htab->hits++;
      if (numchecks > htab->max_scan)
	htab->max_scan = numchecks;
      htab->checks += numchecks;
      return hptr->data;
    }
  }
  if (numchecks > htab->max_scan)
    htab->max_scan = numchecks;
  htab->checks += numchecks;
  return (void *) NULL;
}

/* ---------------------------------------------------------------------------
 * nhashadd: Add a new entry to a numeric hash table.
 */

int
nhashadd(val, hashdata, htab)
    int val, *hashdata;
    NHSHTAB *htab;
{
  int hval;
  NHSHENT *hptr;

  /*
   * Make sure that the entry isn't already in the hash table.  If it
   * is, exit with an error.  Otherwise, create a new hash block and
   * link it in at the head of its thread.
   */

  if (nhashfind(val, htab) != NULL)
    return -1;
  hval = (val & htab->mask);
  htab->entries++;
  if (htab->entry->element[hval] == NULL)
    htab->nulls--;
  hptr = (NHSHENT *) mush_malloc(sizeof(NHSHENT), "nhashent");
  hptr->target = val;
  hptr->data = hashdata;
  hptr->next = htab->entry->element[hval];
  htab->entry->element[hval] = hptr;
  return 0;
}

/* ---------------------------------------------------------------------------
 * nhashdelete: Remove an entry from a numeric hash table.
 */

void
nhashdelete(val, htab)
    int val;
    NHSHTAB *htab;
{
  int hval;
  NHSHENT *hptr, *last;

  hval = (val & htab->mask);
  last = NULL;
  for (hptr = htab->entry->element[hval];
       hptr != NULL;
       last = hptr, hptr = hptr->next) {
    if (val == hptr->target) {
      if (last == NULL)
	htab->entry->element[hval] = hptr->next;
      else
	last->next = hptr->next;
      mush_free(hptr, "nhashent");
      htab->deletes++;
      htab->entries--;
      if (htab->entry->element[hval] == NULL)
	htab->nulls++;
      return;
    }
  }
}

/* ---------------------------------------------------------------------------
 * nhashflush: free all the entries in a hashtable.
 */

void
nhashflush(htab, size)
    NHSHTAB *htab;
    int size;
{
  NHSHENT *hent, *thent;
  int i;

  for (i = 0; i < htab->hashsize; i++) {
    hent = htab->entry->element[i];
    while (hent != NULL) {
      thent = hent;
      hent = hent->next;
      mush_free(thent, "nhashent");
    }
    htab->entry->element[i] = NULL;
  }

  /* Resize if needed.  Otherwise, just zero all the stats */

  if ((size > 0) && (size != htab->hashsize)) {
    mush_free(htab->entry, "nhasnent");
    nhashinit(htab, size);
  } else {
    htab->checks = 0;
    htab->scans = 0;
    htab->max_scan = 0;
    htab->hits = 0;
    htab->entries = 0;
    htab->deletes = 0;
    htab->nulls = htab->hashsize;
  }
}

/* ---------------------------------------------------------------------------
 * nhashrepl: replace the data part of a hash entry.
 */

int
nhashrepl(val, hashdata, htab)
    int val, *hashdata;
    NHSHTAB *htab;
{
  NHSHENT *hptr;
  int hval;

  hval = (val & htab->mask);
  for (hptr = htab->entry->element[hval];
       hptr != NULL;
       hptr = hptr->next) {
    if (hptr->target == val) {
      hptr->data = hashdata;
      return 0;
    }
  }
  return -1;
}
