/* origchat.c */

#include "copyrite.h"
#include "config.h"
#include "conf.h"

#if (CHAT_SYSTEM >= 2)
#include <ctype.h>
#ifdef I_STRING
#include <string.h>
#else
#include <strings.h>
#endif
#ifdef I_STDLIB
#include <stdlib.h>
#endif
#include <math.h>

#include "externs.h"
#include "mushdb.h"
#include "intrface.h"
#include "match.h"
#include "attrib.h"
#include "origchat.h"
#include "confmagic.h"

CHTAB chantab[32];

int nchan _((channel_type channel));
static channel_type find_channel_exact _((char *name));
static int convert_channel_privs _((const char *perms));
static const char *convert_channel_privname _((int privs));
static int add_channel _((const char *name, int privs, int is_quiet, int pos));
void do_cemit _((dbref player, const char *name, const char *msg));
void do_chat _((dbref player, channel_type chan, const char *arg1));
void init_chat _((void));
void do_channel _((dbref player, const char *name, const char *target, const char *com));
void do_chan_admin _((dbref player, char *name, const char *perms, int flag));
void do_channel_list _((dbref player));
channel_type find_channel _((const char *p));
extern void do_channel_who _((dbref player, channel_type chan));

int
nchan(channel)
    channel_type channel;
{
  /* given a flagwise channel, return its index, for chantab purposes */

#ifdef SUN_OS
  return ((int) log2((double) channel));
#else
#ifdef BRAINDEAD_TRUNCATION
  return ((int) (.1 + log((double) channel) / log((double) 2)));
#else
  return ((int) (log((double) channel) / log((double) 2)));
#endif
#endif				/* SUN_OS */
}

channel_type
find_channel(p)
    const char *p;
{
  /* given a channel name, return the flagwise channel */

  int b;
  channel_type c = 0;

  if (!*p)
    return c;

  for (b = 0; !c && (b < 32); b++) {
    if (NChanPrivs(b) != CHP_FORBID) {
      if ((NChanName(b) != NULL) && string_prefix(NChanName(b), p))
	c = 1 << b;
    }
  }

  return c;
}

static channel_type
find_channel_exact(name)
    char *name;
{
  /* exact-match a channel, scanning all channels */
  int b;
  channel_type c = 0;

  for (b = 0; !c && (b < 32); b++) {
    if ((NChanName(b) != NULL) && !strcasecmp(NChanName(b), name))
      c = 1 << b;
  }
  return c;
}

static int
convert_channel_privs(perms)
    const char *perms;
{
  /* given a permission level name, return the permission mask */

  if (string_prefix(perms, "public"))
    return CHP_PUBLIC;
  else if (string_prefix(perms, "admin"))
    return CHP_ADMIN;
  else if (string_prefix(perms, "wizard"))
    return CHP_WIZARD;
#ifdef OBJECT_CHAT
  else if (string_prefix(perms, "object"))
    return CHP_OBJECT;
#endif
  else
    return CHP_FORBID;
}

static const char *
convert_channel_privname(privs)
    int privs;
{
  /* given an integer permission, give permission name */

  switch (privs) {
  case CHP_PUBLIC:
    return "Public";
  case CHP_ADMIN:
    return "Admin";
  case CHP_WIZARD:
    return "Wizard";
#ifdef OBJECT_CHAT
  case CHP_OBJECT:
    return "Object";
#endif
  default:
    return "Forbidden";
  }
}

static int
add_channel(name, privs, is_quiet, pos)
    const char *name;
    int privs;
    int is_quiet;
    int pos;
{
  /* add a channel, return 1 on success, 0 on failure */

  int b;
  int found = 0;

  /* find it */

  if (!pos) {
    /* pick first empty spot, leave first few empty */

    for (b = 4; (b < 32) && !found; b++) {
      if ((chantab[b].name == NULL) && (chantab[b].perms == CHP_FORBID))
	found = 1;
    }

    if (!found)
      return 0;

    b--;			/* we're one ahead of the empty spot */
  } else {
    /* specified a place to put it */
    b = nchan(pos);
  }

  /* add it */
  chantab[b].name = strdup(name);
  chantab[b].perms = privs;
  chantab[b].quiet = is_quiet;

  return 1;
}

void
do_channel(player, name, target, com)
    dbref player;
    const char *name;
    const char *target;
    const char *com;
{
  /* join, quit, wipe, or who a channel */

  channel_type chan;
  int i;
  dbref victim;

#ifndef OBJECT_CHAT
  if (Typeof(player) != TYPE_PLAYER) {
    notify(player, "Sorry, the chat system is only for players.");
    return;
  }
#endif

  if (!name && !*name) {
    notify(player, "You need to specify a channel.");
    return;
  }
  if (!com && !*com) {
    notify(player, "What do you want to do with that channel?");
    return;
  }
  chan = find_channel(name);

  if (!chan) {
    notify(player, "I don't recognize that channel.");
    return;
  }
  if (!ChannelPermit(player, chan)) {
    notify(player, "Sorry, you aren't authorized to do that.");
    return;
  }
  /* The "who" and "wipe" options don't have a target player. */

  if (!strcasecmp("who", com)) {
    do_channel_who(player, chan);
    return;
  } else if (!strcasecmp("wipe", com)) {
    if (!Wizard(player)) {
      notify(player, "Such power is not within you.");
    } else {
      for (i = 0; i < db_top; i++)
	Channels(i) &= ~chan;
      notify(player, "Channel wiped.");
    }
    return;
  }
  /* We can add or delete someone from a channel. Only wizards may
   * do this to someone else.
   */

  /* Determine who is getting added or deleted. If we don't have
   * an argument, we assume it's the player.
   */
  if (!target || !*target)
    victim = player;
  else if (!Wizard(player) ||
	   ((victim = lookup_player(target)) == NOTHING)) {
    notify(player, "Invalid target.");
    return;
  }
  if (!strcasecmp("on", com)) {
    if (!OnChannel(victim, chan)) {
      Channels(victim) |= chan;
      notify(player, "Channel added.");
      if (!QuietChan(chan) && !(Wizard(victim) && Dark(victim)))
	channel_broadcast(chan, "<%s> %s has joined this channel.",
			  channel_name(chan), Name(victim));
    } else {
      notify(player, "That player is already on that channel.");
    }
    return;
  } else if (!strcasecmp("off", com)) {
    if (OnChannel(victim, chan)) {
      Channels(victim) &= ~chan;
      notify(player, "Channel deleted.");
      if (!QuietChan(chan) && !(Wizard(victim) && Dark(victim)))
	channel_broadcast(chan, "<%s> %s has left this channel.",
			  channel_name(chan), Name(victim));
    } else {
      notify(player, "That player is not on that channel.");
    }
    return;
  } else {
    notify(player, "I don't understand what you want to do.");
    return;
  }
}

void
do_chat(player, chan, arg1)
    dbref player;
    channel_type chan;
    const char *arg1;
{
  /* send a message to a channel */

  int key;
  const char *gap;

  if (!chan) {
    notify(player, "I don't recognize that channel.");
    return;
  }
  if (!ChannelPermit(player, chan)) {
    notify(player, "Sorry, you're not authorized to be on that channel.");
    return;
  }
  /* figure out what kind of message we have */
  gap = " ";
  switch (*arg1) {
  case SEMI_POSE_TOKEN:
    gap = "";
  case POSE_TOKEN:
    key = 1;
    arg1 = arg1 + 1;
    break;
  case '\0':
    key = 3;
    break;
  default:
    key = 2;
    break;
  }

  /* now send out the message. If the player isn't on that channel, tell
   * him what he said.
   */
  switch (key) {
  case 1:
    channel_broadcast(chan, "<%s> %s%s%s", channel_name(chan),
		      Name(player), gap, arg1);
    if (!OnChannel(player, chan))
      notify(player, tprintf("To channel %s: %s%s%s", channel_name(chan),
			     Name(player), gap, arg1));
    break;
  case 2:
    channel_broadcast(chan, "<%s> %s says, \"%s\"", channel_name(chan),
		      Name(player), arg1);
    if (!OnChannel(player, chan))
      notify(player, tprintf("To channel %s: %s says, \"%s\"",
			     channel_name(chan), Name(player), arg1));
    break;
  case 3:
    notify(player, "What do you want to say to that channel?");
    break;
  }
}

void
do_cemit(player, name, msg)
    dbref player;
    const char *name;
    const char *msg;
{
  /* Send a message to a channel, no prefix. */

  channel_type chan;

  if (!Can_Cemit(player)) {
    notify(player, "You can't channel-surf that well.");
    return;
  }
  if (!name || !*name || ((chan = find_channel(name)) == 0)) {
    notify(player, "That is not a valid channel.");
    return;
  }
  if (!msg || !*msg) {
    notify(player, "What do you want to emit?");
    return;
  }
  channel_broadcast(chan, "%s", msg);
}

void
do_chan_admin(player, name, perms, flag)
    dbref player;
    char *name;
    const char *perms;
    int flag;
{
  /* wipe, add, remove, rename, change the permissions or quiet status
   * of a channel.
   */

  int chan, privs;
  int i;

  if (!Wizard(player)) {
    notify(player, "Only a wizard may modify channels.");
    return;
  }
  chan = privs = 0;

  if (!name || !*name) {
    notify(player, "You must specify a channel.");
    return;
  }
  if ((flag != 3) && (!perms || !*perms)) {
    notify(player, "What do you want to do with the channel?");
    return;
  }
  switch (flag) {
  case 0:
    /* add a channel */
    /* make sure the channel name is unique */
    if (find_channel(name) != 0) {
      notify(player, "The channel needs a more unique name.");
      return;
    }
    /* get the permissions. Invalid specs default to "forbidden" */
    privs = convert_channel_privs(perms);
    if (privs == CHP_FORBID)
      notify(player, "Warning: channel will be created locked to all.");
    if (add_channel(name, privs, 0, 0))
      notify(player, "Channel created.");
    else
      notify(player, "No more room in the channel table.");
    break;
  case 1:
    /* remove a channel */
    chan = find_channel(name);
    if (!chan) {
      notify(player, "That channel does not appear to exist.");
      return;
    }
    /* remove everyone from the channel */
#ifdef OBJECT_CHAT
    if (ChanPrivs(chan) == CHP_OBJECT)
      for (i = 0; i < db_top; i++)
	if ((Typeof(i) == TYPE_PLAYER) || (Typeof(i) == TYPE_THING))
	  db[i].channels &= ~chan;
#else
    for (i = 0; i < db_top; i++)
      if (Typeof(i) == TYPE_PLAYER)
	db[i].channels &= ~chan;
#endif
    /* zap the channel */
    free(chantab[nchan(chan)].name);
    chantab[nchan(chan)].name = NULL;
    chantab[nchan(chan)].perms = CHP_FORBID;
    notify(player, "Channel removed.");
    break;
  case 2:
    /* rename a channel */
    /* make sure channel exists */
    chan = find_channel_exact(name);
    if (!chan) {
      notify(player, "That channel does not appear to exist.");
      return;
    }
    /* make sure the channel name is unique */
    if (find_channel(perms) != 0) {
      notify(player, "The channel needs a more unique name.");
      return;
    }
    free(chantab[nchan(chan)].name);
    chantab[nchan(chan)].name = strdup(perms);
    notify(player, "Channel renamed.");
    break;
  case 3:
    /* change the permissions on a channel */
    chan = find_channel_exact(name);
    if (!chan) {
      notify(player, "That channel does not appear to exist.");
      return;
    }
    privs = convert_channel_privs(perms);
    if (privs == CHP_FORBID)
      notify(player, "Warning: channel will be locked to all.");
    chantab[nchan(chan)].perms = privs;
    notify(player, "Permissions on channel changed.");
    break;
  case 4:
    if ((chan = find_channel_exact(name)) == 0) {
      notify(player, "That channel does not appear to exist.");
      return;
    }
    if (perms[0] == 'y')
      chantab[nchan(chan)].quiet = 1;
    else if (perms[0] == 'n')
      chantab[nchan(chan)].quiet = 0;
    else {
      notify(player, "Quiet status must be 'yes' or 'no'.");
      return;
    }
    notify(player, "Quiet status changed.");
    break;
  }
}

void
do_channel_list(player)
    dbref player;
{
  int b;

  if (Hasprivs(player)) {
    notify(player, "Channel        Name              Privs        Quiet");

    for (b = 0; b < 32; b++) {
      if ((chantab[b].name == NULL) && (chantab[b].perms == CHP_FORBID))
	continue;
      notify(player, tprintf("0x%-8x     %-15s   %-10s   %s",
			     (int) pow((double) 2, (double) b),
			     chantab[b].name,
			     convert_channel_privname(chantab[b].perms),
			     QuietNChan(b) ? "Yes" : "No"));
    }
  } else {
    notify(player, "Name              Privs        Quiet");

    for (b = 0; b < 32; b++) {
      if (((chantab[b].name == NULL) && (chantab[b].perms == CHP_FORBID))
	  || (chantab[b].perms != CHP_PUBLIC))
	continue;
      notify(player, tprintf("%-15s   %-10s   %s", chantab[b].name,
			     convert_channel_privname(chantab[b].perms),
			     QuietNChan(b) ? "Yes" : "No"));
    }
  }
}


void
init_chat()
{
  /* clear out the channel table and do initialization of defaults. */

  int b;

  for (b = 0; b < 32; b++) {
    chantab[b].name = NULL;
    chantab[b].perms = CHP_FORBID;
  }

  add_channel("Newbie", CHP_PUBLIC, 0, 0x10);
  add_channel("Code", CHP_PUBLIC, 0, 0x20);
  add_channel("Theme", CHP_PUBLIC, 0, 0x40);
  add_channel("OOC", CHP_PUBLIC, 0, 0x80);
  add_channel("Admin", CHP_ADMIN, 0, 0x10000);
  add_channel("Wizard", CHP_WIZARD, 0, 0x1000000);
}

#endif				/* CHAT_SYSTEM */
