/* flags.c */

/* Functions to cope with flags */
#include "config.h"

#ifdef I_SYS_TIME
#include <sys/time.h>
#else
#include <time.h>
#endif
#include <string.h>
#include <stdlib.h>

#include "conf.h"
#include "attrib.h"
#include "mushdb.h"
#include "externs.h"
#include "match.h"
#include "ptab.h"
#include "game.h"
#include "flags.h"
#include "dbdefs.h"
#include "lock.h"
#include "log.h"
#include "confmagic.h"


static FLAG *flag_hash_lookup(const char *name, int type);
int can_set_flag(dbref player, dbref thing, FLAG *flagp, int negate);
static FLAG *letter_to_flagptr(char c, int type, int *toggle);
void decompile_flags(dbref player, dbref thing, const char *name);
void decompile_powers(dbref player, dbref thing, const char *name);

PTAB ptab_flag;

/*   Name       Letter Type            Flag            Perms   Negate_Perm */
FLAG flag_table[] = {

  {"CHOWN_OK", 'C', NOTYPE, CHOWN_OK, F_ANY, F_ANY},
  {"DARK", 'D', NOTYPE, DARK, F_ANY, F_ANY},
  {"GOING", 'G', NOTYPE, GOING, F_INTERNAL, F_ANY},
  {"HAVEN", 'H', NOTYPE, HAVEN, F_ANY, F_ANY},
  {"TRUST", 'I', NOTYPE, INHERIT, F_INHERIT, F_INHERIT},
  {"LINK_OK", 'L', NOTYPE, LINK_OK, F_ANY, F_ANY},
  {"OPAQUE", 'O', NOTYPE, LOOK_OPAQUE, F_ANY, F_ANY},
  {"QUIET", 'Q', NOTYPE, QUIET, F_ANY, F_ANY},
  {"STICKY", 'S', NOTYPE, STICKY, F_ANY, F_ANY},
  {"UNFINDABLE", 'U', NOTYPE, UNFIND, F_ANY, F_ANY},
  {"VISUAL", 'V', NOTYPE, VISUAL, F_ANY, F_ANY},
  {"WIZARD", 'W', NOTYPE, WIZARD, F_INHERIT | F_WIZARD,
   F_INHERIT | F_WIZARD},
  {"SAFE", 'X', NOTYPE, SAFE, F_ANY, F_ANY},
  {"AUDIBLE", 'a', NOTYPE, AUDIBLE, F_ANY, F_ANY},
  {"DEBUG", 'b', NOTYPE, DEBUGGING, F_ANY, F_ANY},
#ifdef USE_WARNINGS
  {"NO_WARN", 'w', NOTYPE, NOWARN, F_ANY, F_ANY},
#endif
  {"ENTER_OK", 'e', NOTYPE, ENTER_OK, F_ANY, F_ANY},
  {"HALT", 'h', NOTYPE, HALT, F_ANY, F_ANY},
  {"NO_COMMAND", 'n', NOTYPE, NO_COMMAND, F_ANY, F_ANY},
  {"LIGHT", 'l', NOTYPE, LIGHT, F_ANY, F_ANY},
#ifdef ROYALTY_FLAG
  {"ROYALTY", 'r', NOTYPE, ROYALTY, F_INHERIT | F_ROYAL,
   F_INHERIT | F_ROYAL},
#endif
  {"TRANSPARENT", 't', NOTYPE, TRANSPARENTED, F_ANY, F_ANY},
  {"VERBOSE", 'v', NOTYPE, VERBOSE, F_ANY, F_ANY},

  {"ANSI", 'A', TYPE_PLAYER, PLAYER_ANSI, F_ANY, F_ANY},

  {"COLOR", 'C', TYPE_PLAYER, PLAYER_COLOR, F_ANY, F_ANY},

  {"MONITOR", 'M', TYPE_PLAYER, PLAYER_MONITOR, F_ROYAL, F_ANY},
  {"NOSPOOF", 'N', TYPE_PLAYER, PLAYER_NOSPOOF, F_ANY | F_ODARK,
   F_ANY | F_ODARK},
  {"SHARED", 'Z', TYPE_PLAYER, PLAYER_ZONE, F_ANY, F_ANY},
  {"CONNECTED", 'c', TYPE_PLAYER, PLAYER_CONNECT, F_INTERNAL | F_MDARK,
   F_INTERNAL | F_MDARK},
  {"GAGGED", 'g', TYPE_PLAYER, PLAYER_GAGGED, F_WIZARD, F_WIZARD},
  {"MYOPIC", 'm', TYPE_PLAYER, PLAYER_MYOPIC, F_ANY, F_ANY},
  {"TERSE", 'x', TYPE_PLAYER, PLAYER_TERSE, F_ANY, F_ANY},
#ifdef JURY_OK
  {"JURY_OK", 'j', TYPE_PLAYER, PLAYER_JURY, F_ROYAL, F_ROYAL},
  {"JUDGE", 'J', TYPE_PLAYER, PLAYER_JUDGE, F_ROYAL, F_ROYAL},
#endif
#ifdef FIXED_FLAG
  {"FIXED", 'F', TYPE_PLAYER, PLAYER_FIXED, F_WIZARD, F_WIZARD},
#endif
#ifdef ONLINE_REG
  {"UNREGISTERED", '?', TYPE_PLAYER, PLAYER_UNREG, F_ROYAL, F_ROYAL},
#endif
#ifdef VACATION_FLAG
  {"ON-VACATION", 'o', TYPE_PLAYER, PLAYER_VACATION, F_ANY, F_ANY},
#endif
  {"SUSPECT", 's', TYPE_PLAYER, PLAYER_SUSPECT, F_WIZARD | F_MDARK,
   F_WIZARD | F_MDARK},
  {"PARANOID", 'p', TYPE_PLAYER, PLAYER_PARANOID, F_ANY | F_ODARK,
   F_ANY | F_ODARK},
  {"NOACCENTS", '~', TYPE_PLAYER, PLAYER_NOACCENTS, F_ANY, F_ANY},

  {"MONITOR", 'M', TYPE_THING, THING_LISTEN, F_ANY, F_ANY},
  {"DESTROY_OK", 'd', TYPE_THING, THING_DEST_OK, F_ANY, F_ANY},
  {"PUPPET", 'p', TYPE_THING, THING_PUPPET, F_ANY, F_ANY},
  {"NO_LEAVE", 'N', TYPE_THING, THING_NOLEAVE, F_ANY, F_ANY},
  {"LISTEN_PARENT", '^', TYPE_THING, THING_INHEARIT, F_ANY, F_ANY},
  {"Z_TEL", 'Z', TYPE_THING, THING_Z_TEL, F_ANY, F_ANY},

  {"ABODE", 'A', TYPE_ROOM, ROOM_ABODE, F_ANY, F_ANY},
  {"FLOATING", 'F', TYPE_ROOM, ROOM_FLOATING, F_ANY, F_ANY},
  {"JUMP_OK", 'J', TYPE_ROOM, ROOM_JUMP_OK, F_ANY, F_ANY},
  {"MONITOR", 'M', TYPE_ROOM, ROOM_LISTEN, F_ANY, F_ANY},
  {"LISTEN_PARENT", '^', TYPE_ROOM, ROOM_INHEARIT, F_ANY, F_ANY},
  {"Z_TEL", 'Z', TYPE_ROOM, ROOM_Z_TEL, F_ANY, F_ANY},
  {"NO_TEL", 'N', TYPE_ROOM, ROOM_NO_TEL, F_ANY, F_ANY},
#ifdef UNINSPECTED_FLAG
  {"UNINSPECTED", 'u', TYPE_ROOM, ROOM_UNINSPECT, F_ROYAL, F_ROYAL},
#endif

  {"CLOUDY", 'x', TYPE_EXIT, EXIT_CLOUDY, F_ANY, F_ANY},

  {"MARKED", '\0', NOTYPE, MARKED, F_INTERNAL | F_DARK,
   F_INTERNAL | F_DARK},
  {"GOING_TWICE", '\0', NOTYPE, GOING_TWICE, F_INTERNAL | F_DARK,
   F_INTERNAL | F_DARK},
  {NULL, '\0', 0, 0, 0, 0}
};

FLAG type_table[] = {
  {"PLAYER", 'P', TYPE_PLAYER, TYPE_PLAYER, F_INTERNAL,
   F_INTERNAL},
  {"ROOM", 'R', TYPE_ROOM, TYPE_ROOM, F_INTERNAL,
   F_INTERNAL},
  {"EXIT", 'E', TYPE_EXIT, TYPE_EXIT, F_INTERNAL,
   F_INTERNAL},
  {"THING", 'T', TYPE_THING, TYPE_THING, F_INTERNAL,
   F_INTERNAL},
  {NULL, '\0', 0, 0, 0, 0}
};

static FLAG_ALIAS flag_alias_tab[] = {
  {"INHERIT", "TRUST"},
  {"TRACE", "DEBUG"},
#ifdef USE_WARNINGS
  {"NOWARN", "NO_WARN"},
#endif
  {"NOCOMMAND", "NO_COMMAND"},
  {"LISTENER", "MONITOR"},
  {"WATCHER", "MONITOR"},
  {"ZONE", "SHARED"},
  {"COLOUR", "COLOR"},
#ifdef JURY_OK
  {"JURYOK", "JURY_OK"},
#endif
#ifdef VACATION_FLAG
  {"VACATION", "ON-VACATION"},
#endif
  {"DEST_OK", "DESTROY_OK"},
  {"NOLEAVE", "NO_LEAVE"},
  {"TEL_OK", "JUMP_OK"},
  {"TELOK", "JUMP_OK"},
  {"TEL-OK", "JUMP_OK"},
  {"^", "LISTEN_PARENT"},

  {NULL, NULL}
};

/*   Name      Flag   */
POWER power_table[] = {
  {"Announce", CAN_WALL},
  {"Boot", CAN_BOOT},
  {"Builder", CAN_BUILD},
  {"Cemit", CEMIT},
  {"Chat_Privs", CHAT_PRIVS},
  {"Functions", GLOBAL_FUNCS},
  {"Guest", IS_GUEST},
  {"Halt", HALT_ANYTHING},
  {"Hide", CAN_HIDE},
  {"Idle", UNLIMITED_IDLE},
  {"Immortal", NO_PAY | NO_QUOTA | UNKILLABLE},
  {"Link_Anywhere", LINK_ANYWHERE},
  {"Login", LOGIN_ANYTIME},
  {"Long_Fingers", LONG_FINGERS},
  {"No_Pay", NO_PAY},
  {"No_Quota", NO_QUOTA},
  {"Open_Anywhere", OPEN_ANYWHERE},
  {"Pemit_All", PEMIT_ALL},
  {"Player_Create", CREATE_PLAYER},
  {"Poll", SET_POLL},
  {"Queue", HUGE_QUEUE},
  {"Quotas", CHANGE_QUOTAS},
  {"Search", SEARCH_EVERYTHING},
  {"See_All", SEE_ALL},
  {"See_Queue", PS_ALL},
  {"Tport_Anything", TEL_OTHER},
  {"Tport_Anywhere", TEL_ANYWHERE},
  {"Unkillable", UNKILLABLE},
  {NULL, 0}
};

static POWER_ALIAS power_alias_tab[] = {
  {"@cemit", "Cemit"},
  {"@wall", "Announce"},
  {"wall", "Announce"},
  {NULL, NULL}
};




/*---------------------------------------------------------------------------
 * Flag hash table handlers 
 */


static FLAG *
flag_hash_lookup(const char *name, int type)
{
  FLAG *f;

  f = (FLAG *) ptab_find(&ptab_flag, name);
  if (f)
    return f;

  /* If the name is a single character, search the flag characters */
  if (name && *name && !*(name + 1)) {
    f = ptab_firstentry(&ptab_flag);
    while (f) {
      /* Doh! Kludge-city. We'll ignore the CHOWN_OK flag on players, because
       * it's more useful to check 'C' as COLOR. Argle.
       */
      if ((*name == f->letter) &&
	  ((f->type == type) ||
	   ((f->type == NOTYPE) && !((*name == 'C') && (type == TYPE_PLAYER)))))
	return f;
      f = ptab_nextentry(&ptab_flag);
    }
  }

  /* provided for backwards compatibility: type flag checking */
  for (f = type_table; f->name != NULL; f++)
    if (string_prefix(name, f->name))
      return f;

  return NULL;
}

/* This is a stub function to add a flag. It performs no error-checking,
 * so it's up to you to be sure you're adding a flag that's properly
 * set up and that'll work ok
 */
void
flag_add(const char *name, FLAG *f)
{
  ptab_start_inserts(&ptab_flag);
  ptab_insert(&ptab_flag, name, f);
  ptab_end_inserts(&ptab_flag);
}

void
init_flag_table()
{
  FLAG *f;
  FLAG_ALIAS *a;

  ptab_init(&ptab_flag);

  /* do regular flags first */
  for (f = flag_table; f->name; f++)
    flag_add(f->name, f);

  /* now add in the aliases */
  for (a = flag_alias_tab; a->alias; a++) {
    for (f = flag_table; f->name; f++) {
      if (strcmp(a->realname, f->name) == 0) {
	flag_add(a->alias, f);
	break;
      }
    }
    if (!f)
      do_rawlog(LT_ERR,
		T("FLAG INIT: flag alias %s matches no known flag."), a->alias);
  }
}

/*---------------------------------------------------------------------------
 * Other functions dealing with flags
 */

int
can_set_flag(dbref player, dbref thing, FLAG *flagp, int negate)
{
  /* returns 1 if player can set a flag on thing. */
  int myperms;

  if (!flagp || !GoodObject(player) || !GoodObject(thing))
    return 0;

  myperms = negate ? flagp->negate_perms : flagp->perms;

  if (myperms & F_INTERNAL)
    return 0;

  if ((flagp->type != NOTYPE) && (flagp->type != Typeof(thing)))
    return 0;

  if ((myperms & F_INHERIT) && !Wizard(player) &&
      (!Inheritable(player) || !Owns(player, thing)))
    return 0;

  /* You've got to *own* something (or be Wizard) to set it
   * chown_ok or dest_ok. This prevents subversion of the
   * zone-restriction on @chown and @dest
   */
  if (((flagp->flag == CHOWN_OK) && (flagp->type == NOTYPE)) ||
      ((flagp->flag == THING_DEST_OK) && (flagp->type == TYPE_THING))) {
    if (!Owns(player, thing) && !Wizard(player))
      return 0;
    else
      return 1;
  }
  /* Checking for the ZONE flag. If you set this, the player had
   * better be zone-locked! 
   */
  if ((flagp->type == TYPE_PLAYER) && (flagp->flag == PLAYER_ZONE) &&
      !negate && (getlock(thing, Zone_Lock) == TRUE_BOOLEXP)) {
    notify(player, T("You must @lock/zone before you can set a player ZONE"));
    return 0;
  }
  if (myperms & F_ANY)
    return 1;
  if ((myperms & F_WIZARD) && !Wizard(player))
    return 0;
  else if ((myperms & F_ROYAL) && !Hasprivs(player))
    return 0;
  else if ((myperms & F_GOD) && !God(player))
    return 0;

  /* special case for the privileged flags. We do need to check
   * for royalty setting flags on objects they don't own, because
   * the controls check will not stop the flag set if the royalty
   * player is in a zone. Also, only God can set the wizbit on
   * players.
   */
  if (Wizard(thing) && (flagp->flag == PLAYER_GAGGED) &&
      (flagp->type == TYPE_PLAYER))
    return 0;			/* can't gag wizards/God */

  if (God(player))		/* God can do (almost) anything) */
    return 1;

  /* Make sure we don't accidentally permission-check toggles when
   * checking priv bits.
   */
  if (flagp->type == NOTYPE) {

    /* A wiz can set things he owns WIZ, but nothing else. */
    if ((flagp->flag == WIZARD) && !negate)
      return (Wizard(player) && Owns(player, thing) && !IsPlayer(thing));
    /* A wiz can unset the WIZ bit on any non-player */
    if ((flagp->flag == WIZARD) && negate)
      return (Wizard(player) && !IsPlayer(thing));
    /* Wizards can set or unset anything royalty. Royalty can set anything
     * they own royalty, but cannot reset their own bits. */
#ifdef ROYALTY_FLAG
    if (flagp->flag == ROYALTY) {
      return (!Guest(thing) && (Wizard(player) || (Royalty(player) &&
						   Owns(player, thing)
						   && !IsPlayer(thing))));
    }
#endif
  }
  return 1;
}

const char *
unparse_flags(dbref thing, dbref player)
{
  /* print out the flag symbols (letters) */

  static char buf[BUFFER_LEN];
  char *p;
  static const char type_codes[] = "RTEP--";
  FLAG *f;
  int t;
  int flag_need = Flags(thing);
  int toggle_need = Toggles(thing);

  p = buf;
  t = Typeof(thing);

  if (t == TYPE_GARBAGE) {
    *p = '\0';
    return buf;
  }

  *p++ = type_codes[t];

  f = ptab_firstentry(&ptab_flag);
  while (f) {
    if ((f->type == t) && (toggle_need & f->flag) &&
	Can_See_Flag(player, thing, f)) {
      toggle_need &= ~f->flag;
      *p++ = f->letter;
    } else if ((f->type == NOTYPE) && (flag_need & f->flag) &&
	       Can_See_Flag(player, thing, f)) {
      flag_need &= ~f->flag;
      *p++ = f->letter;
    }
    f = ptab_nextentry(&ptab_flag);
  }

  *p = '\0';

  return buf;
}

const char *
flag_description(dbref player, dbref thing)
{
  static char buf[BUFFER_LEN];
  char *bp;
  FLAG *f;
  int t = Typeof(thing);
  int flag_need = Flags(thing);
  int toggle_need = Toggles(thing);

  bp = buf;
  safe_str(T("Type: "), buf, &bp);
  switch (t) {
  case TYPE_ROOM:
    safe_str("Room", buf, &bp);
    break;
  case TYPE_EXIT:
    safe_str("Exit", buf, &bp);
    break;
  case TYPE_THING:
    safe_str("Thing", buf, &bp);
    break;
  case TYPE_PLAYER:
    safe_str("Player", buf, &bp);
    break;
  default:
    safe_str("***UNKNOWN TYPE***", buf, &bp);
    break;
  }
  safe_str(T(" Flags:"), buf, &bp);

  f = ptab_firstentry(&ptab_flag);
  while (f) {
    if ((f->type == t) && (toggle_need & f->flag) &&
	Can_See_Flag(player, thing, f)) {
      safe_chr(' ', buf, &bp);
      safe_str(f->name, buf, &bp);
      toggle_need &= ~f->flag;
    }
    if ((f->type == NOTYPE) && (flag_need & f->flag) &&
	Can_See_Flag(player, thing, f)) {
      safe_chr(' ', buf, &bp);
      safe_str(f->name, buf, &bp);
      flag_need &= ~f->flag;
    }
    f = ptab_nextentry(&ptab_flag);
  }

  *bp = '\0';

  return buf;
}


/* Used to show the default toggle configuration from do_config_list
 * 'mask' here covers the NOTYPE flags. The others we have to
 * get from the appropriate part of the options structure
 */
const char *
togglemask_to_string(int type, object_flag_type mask)
{
  static char fbuf[BUFFER_LEN];
  char *bp;
  FLAG *f;
  object_flag_type toggle = 0;

  switch (type) {
  case TYPE_THING:
    toggle = options.thing_toggles;
    break;
  case TYPE_PLAYER:
    toggle = options.player_toggles;
    break;
  case TYPE_EXIT:
    toggle = options.exit_toggles;
    break;
  case TYPE_ROOM:
    toggle = options.room_toggles;
    break;
  }
  bp = fbuf;
  f = ptab_firstentry(&ptab_flag);
  while (f) {
    if (((f->type == NOTYPE) && (mask & f->flag)) ||
	((f->type == type) && (toggle & f->flag))) {
      if (bp != fbuf)
	safe_chr(' ', fbuf, &bp);
      safe_str(f->name, fbuf, &bp);
      if (f->type == NOTYPE)
	mask &= ~f->flag;
      else if (f->type == type)
	toggle &= ~f->flag;
    }
    f = ptab_nextentry(&ptab_flag);
  }
  *bp = '\0';
  return fbuf;
}

static FLAG *
letter_to_flagptr(char c, int type, int *toggle)
{
  /* convert letter to flag */

  FLAG *f;

  f = ptab_firstentry(&ptab_flag);
  while (f) {
    /* Doh! Kludge-city. We'll ignore the CHOWN_OK flag on players, because
     * it's more useful to check 'C' as COLOR. Argle.
     */
    if ((f->letter == c) && (f->type == type)) {
      *toggle = 1;
      return f;
    } else if ((f->letter == c) && (f->type == NOTYPE) &&
	       !((c == 'C') && (type == TYPE_PLAYER))) {
      *toggle = 0;
      return f;
    }
    f = ptab_nextentry(&ptab_flag);
  }

  /* Do we need to do this? */
  *toggle = 1;
  return NULL;
}


object_flag_type
find_flag(const char *name, int type, int *toggle, int is_conf)
{
  /* given a flag name and an object type, return a flag, and set the
   * value of toggle to 0 if flag, 1 if toggle. We also check if it's
   * a legal flag to set, if it's a conf flag set.
   */

  FLAG *f;

  *toggle = 0;

  if ((f = flag_hash_lookup(name, type)) == NULL)
    return -1;

  if (is_conf && (f->perms == F_INTERNAL))
    return -1;

  if (f->type == NOTYPE) {
    return f->flag;
  } else {
    *toggle = 1;
    if (type != f->type) {
      if (is_conf)
	return -2;
      else
	return -1;
    } else {
      return f->flag;
    }
  }
  return f->flag;		/* NOTREACHED */
}

void
decompile_flags(dbref player, dbref thing, const char *name)
{
  /* print out the flags for a decompile */

  FLAG *f;
  int t = Typeof(thing);
  int flag_need = Flags(thing);
  int toggle_need = Toggles(thing);

  f = ptab_firstentry(&ptab_flag);
  while (f) {
    if ((f->type == t) && (toggle_need & f->flag) &&
	Can_See_Flag(player, thing, f)) {
      notify_format(player, "@set %s = %s", name, f->name);
      toggle_need &= ~f->flag;
    } else if ((f->type == NOTYPE) && (flag_need & f->flag) &&
	       !(f->perms & F_INTERNAL) && Can_See_Flag(player, thing, f)) {
      notify_format(player, "@set %s = %s", name, f->name);
      flag_need &= ~f->flag;
    }
    f = ptab_nextentry(&ptab_flag);
  }
}


void
decompile_powers(dbref player, dbref thing, const char *name)
{
  /* print out the powers for a decompile */

  POWER *p;

  for (p = power_table; p->name; p++) {
    /* Special case for immortal, which we don't show any more */
    if (!strcasecmp(p->name, "immortal"))
      continue;
    if (Powers(thing) & p->flag)
      notify_format(player, "@power %s = %s", name, p->name);
  }
}

/*
 * Extended to allow all multi-type toggles
 */
static FLAG hack_table[] = {
  {"MONITOR", 'M', TYPE_PLAYER, PLAYER_MONITOR, F_ROYAL, F_ROYAL},
  {"MONITOR", 'M', TYPE_THING, THING_LISTEN, F_ANY, F_ANY},
  {"MONITOR", 'M', TYPE_ROOM, ROOM_LISTEN, F_ANY, F_ANY},
  {"LISTEN_PARENT", '^', TYPE_THING, THING_INHEARIT, F_ANY, F_ANY},
  {"LISTEN_PARENT", '^', TYPE_ROOM, ROOM_INHEARIT, F_ANY, F_ANY},
  {"Z_TEL", 'Z', TYPE_THING, THING_Z_TEL, F_ANY, F_ANY},
  {"Z_TEL", 'Z', TYPE_ROOM, ROOM_Z_TEL, F_ANY, F_ANY},
  {NULL, '\0', 0, 0, 0, 0}
};

void
set_flag(dbref player, dbref thing, const char *flag, int negate, int hear,
	 int listener)
{
  /* attempt to set a flag on an object */

  FLAG *f, *o;
  char tbuf1[BUFFER_LEN];
  char *tp;

  if ((f = flag_hash_lookup(flag, Typeof(thing))) == NULL) {
    notify(player, T("I don't recognize that flag."));
    return;
  }
  /* HORRIBLE HACK: added to make MONITOR work. This needs to
   * be fixed in the future, _somehow_...
   */
  if (!strcmp(f->name, "MONITOR") || !strcmp(f->name, "LISTEN_PARENT") ||
      !strcmp(f->name, "Z_TEL")) {
    o = f;
    for (f = hack_table; f->name != NULL; f++)
      if ((Typeof(thing) == f->type) && !strcmp(f->name, o->name))
	break;
    if (f->name == NULL) {
      notify(player, T("Permission denied."));
      return;
    }
  }

  if (!can_set_flag(player, thing, f, negate)) {
    notify(player, T("Permission denied."));
    return;
  }
  /* The only players who can be Dark are wizards. */
  if ((f->flag == DARK) && !negate && (f->type == NOTYPE) &&
      Alive(thing) && !Wizard(thing)) {
    notify(player, T("Permission denied."));
    return;
  }
  if (negate) {

    if ((f->flag == GOING) && (f->type == NOTYPE)) {
      /* This is, frankly, a bit of a hack. */
      notify(player,
	     T("@set !GOING has been disabled. Use @undestroy instead."));
      return;
    }
    /* remove the flag */
    if (f->type == NOTYPE)
      Flags(thing) &= ~(f->flag);
    else
      Toggles(thing) &= ~(f->flag);
    /* log if necessary */
    if ((f->flag == WIZARD) && (f->type == NOTYPE))
      do_log(LT_WIZ, player, thing, "WIZFLAG RESET");
#ifdef ROYALTY_FLAG
    else if ((f->flag == ROYALTY) && (f->type == NOTYPE))
      do_log(LT_WIZ, player, thing, "ROYAL FLAG RESET");
#endif
    else if ((f->flag == PLAYER_SUSPECT) && (f->type == TYPE_PLAYER))
      do_log(LT_WIZ, player, thing, "SUSPECT FLAG RESET");
    /* those who unDARK return to the WHO */
    if ((f->flag == DARK) && (f->type == NOTYPE) && (IsPlayer(thing)))
      hide_player(thing, 0);
    /* notify the area if something stops listening, but only if it
       wasn't listening before */
    if (IsThing(thing) && (f->type == TYPE_THING) &&
	GoodObject(Location(thing)) && (hear || listener) &&
	!Hearer(thing) && !Listener(thing)) {
      tp = tbuf1;
      safe_format(tbuf1, &tp, T("%s is no longer listening."), Name(thing));
      *tp = '\0';
      notify_except(Contents(Location(thing)), NOTHING, tbuf1);
      notify_except(Contents(thing), NOTHING, tbuf1);
    }
    if (IsRoom(thing) && (f->type == TYPE_ROOM) &&
	(f->flag == ROOM_LISTEN) && !hear && !Listener(thing)) {
      tp = tbuf1;
      safe_format(tbuf1, &tp, T("%s is no longer listening."), Name(thing));
      *tp = '\0';
      notify_except(Contents(thing), NOTHING, tbuf1);
    }
    if ((f->flag == AUDIBLE) && (f->type == NOTYPE)) {
      switch (Typeof(thing)) {
      case TYPE_EXIT:
	if (Audible(Source(thing))) {
	  tp = tbuf1;
	  safe_format(tbuf1, &tp, T("Exit %s is no longer broadcasting."),
		      Name(thing));
	  *tp = '\0';
	  notify_except(Contents(Source(thing)), NOTHING, tbuf1);
	}
	break;
      case TYPE_ROOM:
	notify_except(Contents(thing), NOTHING,
		      T("Audible exits in this room have been deactivated."));
	break;
      case TYPE_THING:
      case TYPE_PLAYER:
	notify_except(Contents(thing), thing,
		      T("This room is no longer broadcasting."));
	notify(thing, T("Your contents can no longer be heard from outside."));
	break;
      }
    }
    if (((f->flag == QUIET) && (f->type == NOTYPE)) ||
	(!AreQuiet(player, thing))) {
      tp = tbuf1;
      safe_str(Name(thing), tbuf1, &tp);
      safe_str(" - ", tbuf1, &tp);
      safe_str(f->name, tbuf1, &tp);
      safe_str(T(" reset."), tbuf1, &tp);
      *tp = '\0';
      notify(player, tbuf1);
    }
  } else {

    /* set the flag */
    if (f->type == NOTYPE)
      Flags(thing) |= f->flag;
    else
      Toggles(thing) |= f->flag;
    /* log if necessary */
    if ((f->flag == WIZARD) && (f->type == NOTYPE))
      do_log(LT_WIZ, player, thing, "WIZFLAG SET");
#ifdef ROYALTY_FLAG
    if ((f->flag == ROYALTY) && (f->type == NOTYPE))
      do_log(LT_WIZ, player, thing, "ROYAL FLAG SET");
#endif
    else if ((f->flag == PLAYER_SUSPECT) && (f->type == TYPE_PLAYER))
      do_log(LT_WIZ, player, thing, "SUSPECT FLAG SET");
    if ((f->type == NOTYPE) && (f->flag == INHERIT)
	&& GoodObject(Zone(thing)))
      notify(player, T("Warning: Setting inherit flag on zoned object"));
    /* DARK players should be treated as logged out */
    if ((f->flag == DARK) && (f->type == NOTYPE) && (IsPlayer(thing)))
      hide_player(thing, 1);
    /* notify area if something starts listening */
    if (IsThing(thing) && (f->type == TYPE_THING) &&
	GoodObject(Location(thing)) &&
	((f->flag == THING_PUPPET) || (f->flag == THING_LISTEN)) &&
	!hear && !listener) {
      tp = tbuf1;
      safe_format(tbuf1, &tp, T("%s is now listening."), Name(thing));
      *tp = '\0';
      notify_except(Contents(Location(thing)), NOTHING, tbuf1);
      notify_except(Contents(thing), NOTHING, tbuf1);
    }
    if (IsRoom(thing) && (f->type == TYPE_ROOM) &&
	(f->flag == ROOM_LISTEN) && !hear && !listener) {
      tp = tbuf1;
      safe_format(tbuf1, &tp, T("%s is now listening."), Name(thing));
      *tp = '\0';
      notify_except(Contents(thing), NOTHING, tbuf1);
    }
    /* notify for audible exits */
    if ((f->flag == AUDIBLE) && (f->type == NOTYPE)) {
      switch (Typeof(thing)) {
      case TYPE_EXIT:
	if (Audible(Source(thing))) {
	  tp = tbuf1;
	  safe_format(tbuf1, &tp, T("Exit %s is now broadcasting."),
		      Name(thing));
	  *tp = '\0';
	  notify_except(Contents(Source(thing)), NOTHING, tbuf1);
	}
	break;
      case TYPE_ROOM:
	notify_except(Contents(thing), NOTHING,
		      T("Audible exits in this room have been activated."));
	break;
      case TYPE_PLAYER:
      case TYPE_THING:
	notify_except(Contents(thing), thing,
		      T("This room is now broadcasting."));
	notify(thing, T("Your contents can now be heard from outside."));
	break;
      }
    }
    if (((f->flag == QUIET) && (f->flag == NOTYPE)) ||
	(!AreQuiet(player, thing))) {
      tp = tbuf1;
      safe_str(Name(thing), tbuf1, &tp);
      safe_str(" - ", tbuf1, &tp);
      safe_str(f->name, tbuf1, &tp);
      safe_str(T(" set."), tbuf1, &tp);
      *tp = '\0';
      notify(player, tbuf1);
    }
  }
}


int
flaglist_check(player, it, fstr, type)
    dbref player;
    dbref it;
    const char *fstr;
    int type;			/* 0 for orflags, 1 for andflags */
{
  char *s;
  FLAG *fp;
  int toggle, negate, temp;
  int ret = type;

  toggle = negate = temp = 0;
  if (it == NOTHING)
    return 0;
  for (s = (char *) fstr; *s; s++) {

    /* Check for a negation sign. If we find it, we note it and 
     * increment the pointer to the next character.
     */

    if (*s == '!') {
      negate = 1;
      s++;
    } else {
      negate = 0;		/* It's important to clear this at appropriate times;
				 * else !Dc means (!D && !c), instead of (!D && c). */
    }

    if (!*s)
      /* We got a '!' that wasn't followed by a letter.
       * Fail the check. */
      return (type == 1) ? 0 : ret;
    /* Find the flag. */
    if ((fp = letter_to_flagptr(*s, Typeof(it), &toggle)) == NULL) {

      /* Maybe *s is a type specifier (P, T, E, R). These aren't really
       * flags, but we grandfather them in to preserve old code
       */

      if ((*s == 'T') || (*s == 'R') || (*s == 'E') || (*s == 'P')) {
	temp = (*s == 'T') ? (Typeof(it) == TYPE_THING) :
	  ((*s == 'R') ? (Typeof(it) == TYPE_ROOM) :
	   ((*s == 'E') ? (Typeof(it) == TYPE_EXIT) :
	    (Typeof(it) == TYPE_PLAYER)));
	if ((type == 1) && ((negate && temp) || (!negate && !temp)))
	  return 0;
	else if ((type == 0) && ((!negate && temp) || (negate && !temp)))
	  ret |= 1;
      } else {
	/* Either we got a '!' that wasn't followed by a letter, or
	 * we couldn't find that flag. For AND, since we've failed
	 * a check, we can return false. Otherwise we just go on.
	 */

	if (type == 1)
	  return 0;
	else
	  continue;
      }
    } else {

      /* does the object have this flag? */

      if ((!toggle && (Flags(it) & fp->flag)) ||
	  (toggle && (Toggles(it) & fp->flag) && Can_See_Flag(player, it, fp)))
	temp = 1;
      else
	temp = 0;
      if ((type == 1) && ((negate && temp) || (!negate && !temp))) {

	/* Too bad there's no NXOR function...
	 * At this point we've either got a flag and we don't want
	 * it, or we don't have a flag and we want it. Since it's
	 * AND, we return false.
	 */
	return 0;
      } else if ((type == 0) && ((!negate && temp) || (negate && !temp))) {

	/* We've found something we want, in an OR. We OR a
	 * true with the current value.
	 */

	ret |= 1;
      }
      /* Otherwise, we don't need to do anything. */
    }
  }
  return (ret);
}


int
sees_flag(dbref privs, dbref thing, const char *name)
{
  /* Does thing have the flag named name && can privs see it? */
  FLAG *f, *o;
  int retval;
  if ((f = flag_hash_lookup(name, Typeof(thing))) == NULL)
    return 0;
  /* HORRIBLE HACK: added to make MONITOR work. This needs to
   * be fixed in the future, _somehow_...
   */
  if (!strcmp(f->name, "MONITOR")
      || !strcmp(f->name, "LISTEN_PARENT")
      || !strcmp(f->name, "Z_TEL")) {
    o = f;
    for (f = hack_table; f->name != NULL; f++)
      if ((Typeof(thing) == f->type)
	  && !strcmp(f->name, o->name))
	break;
    if (f->name == NULL) {
      return 0;
    }
    retval = Toggles(thing) & f->flag;
  } else if (f->type == NOTYPE) {
    retval = Flags(thing) & f->flag;
  } else {
    if (Typeof(thing) != f->type)
      return 0;
    else
      retval = Toggles(thing) & f->flag;
  }
  return retval && Can_See_Flag(privs, thing, f);
}


const char *
power_description(dbref thing)
{
  static char fbuf[BUFFER_LEN];
  char *bp;
  POWER *p;
  bp = fbuf;
  for (p = power_table; p->name; p++) {
    /* Special case for immortal, which we don't show any more */
    if (!strcasecmp(p->name, "immortal"))
      continue;
    if (Powers(thing) & p->flag) {
      if (bp != fbuf)
	safe_chr(' ', fbuf, &bp);
      safe_str(p->name, fbuf, &bp);
    }
  }
  *bp = '\0';
  return fbuf;
}

object_flag_type
find_power(const char *name)
{
  POWER *p;
  POWER_ALIAS *a;
  for (p = power_table; p->name; p++) {
    if (string_prefix(p->name, name))
      return p->flag;
  }

  /* Check the alias table */
  for (a = power_alias_tab; a->alias; a++) {
    if (string_prefix(a->alias, name))
      return find_power(a->realname);
  }

  /* Got nothing. Return -1 */
  return -1;
}

const char *
show_command_flags(int flags, int powers)
{
  static char fbuf[BUFFER_LEN];
  char *bp;
  POWER *p;
  FLAG *f;

  bp = fbuf;
  if (powers) {
    safe_str("Powers     : ", fbuf, &bp);
    for (p = power_table; p->name; p++)
      if (powers & p->flag) {
	safe_str(p->name, fbuf, &bp);
	safe_chr(' ', fbuf, &bp);
      }
  }

  /* do generic flags */
  if (flags) {
    if (powers)
      safe_chr('\n', fbuf, &bp);
    safe_str("Flags      : ", fbuf, &bp);
    f = ptab_firstentry(&ptab_flag);
    while (f) {
      if ((f->type == NOTYPE) && (flags & f->flag)) {
	safe_str(f->name, fbuf, &bp);
	safe_chr(' ', fbuf, &bp);
      }
      f = ptab_nextentry(&ptab_flag);
    }
  }


  *bp = '\0';
  return fbuf;
}

int
match_flag(const char *name)
{
  FLAG *t;
  t = (FLAG *) ptab_find(&ptab_flag, name);
  return ((t && (t->type == NOTYPE)) ? t->flag : 0);
}

int
match_power(const char *name)
{
  object_flag_type p;
  p = find_power(name);
  return ((p > -1) ? p : 0);
}
