/* htab.c - table hashing routines */

/* This code is ripped out of TinyMUSH 2.2. */
/* Minor tweaking to make in Penn-compatible by Trivian (xeno@mix.hive.no) */


#include "config.h"
#include "copyrite.h"
#include <string.h>
#include "conf.h"
#include "externs.h"

#include "htab.h"
#include "memcheck.h"
#include "mymalloc.h"
#include "confmagic.h"

/* ---------------------------------------------------------------------------
 * hash_val: Compute hash value of a string for a hash table.
 */
/*#define NEW_HASH_FUN /**/
#ifdef NEW_HASH_FUN

/* This hash function adapted from http://burtleburtle.net/bob/hash/evahash.html */

typedef unsigned long int u4;	/* unsigned 4-byte type */
typedef unsigned char u1;	/* unsigned 1-byte type */

/* The mixing step */
#define mix(a,b,c) \
{ \
  a=a-b;  a=a-c;  a=a^(c>>13); \
  b=b-c;  b=b-a;  b=b^(a<<8);  \
  c=c-a;  c=c-b;  c=c^(b>>13); \
  a=a-b;  a=a-c;  a=a^(c>>12); \
  b=b-c;  b=b-a;  b=b^(a<<16); \
  c=c-a;  c=c-b;  c=c^(b>>5);  \
  a=a-b;  a=a-c;  a=a^(c>>3);  \
  b=b-c;  b=b-a;  b=b^(a<<10); \
  c=c-a;  c=c-b;  c=c^(b>>15); \
}

/* The whole new hash function */
int
hash_val(k, mask)
    register const char *k;	/* the key */
    int mask;
{
  register u4 a, b, c;		/* the internal state */
  u4 len, length;		/* how many key bytes still need mixing */
  static u4 initval = 5432;	/* the previous hash, or an arbitrary value */

  /* Set up the internal state */
  length = len = strlen(k);
  a = b = 0x9e3779b9;		/* the golden ratio; an arbitrary value */
  c = initval;			/* variable initialization of internal state */

   /*---------------------------------------- handle most of the key */
  while (len >= 12) {
    a = a + (k[0] + ((u4) k[1] << 8) + ((u4) k[2] << 16) + ((u4) k[3] << 24));
    b = b + (k[4] + ((u4) k[5] << 8) + ((u4) k[6] << 16) + ((u4) k[7] << 24));
    c = c + (k[8] + ((u4) k[9] << 8) + ((u4) k[10] << 16) + ((u4) k[11] << 24));
    mix(a, b, c);
    k = k + 12;
    len = len - 12;
  }

   /*------------------------------------- handle the last 11 bytes */
  c = c + length;
  switch (len) {		/* all the case statements fall through */
  case 11:
    c = c + ((u4) k[10] << 24);
  case 10:
    c = c + ((u4) k[9] << 16);
  case 9:
    c = c + ((u4) k[8] << 8);
    /* the first byte of c is reserved for the length */
  case 8:
    b = b + ((u4) k[7] << 24);
  case 7:
    b = b + ((u4) k[6] << 16);
  case 6:
    b = b + ((u4) k[5] << 8);
  case 5:
    b = b + k[4];
  case 4:
    a = a + ((u4) k[3] << 24);
  case 3:
    a = a + ((u4) k[2] << 16);
  case 2:
    a = a + ((u4) k[1] << 8);
  case 1:
    a = a + k[0];
    /* case 0: nothing left to add */
  }
  mix(a, b, c);
   /*-------------------------------------------- report the result */
  return c & mask;
}


#else				/* NEW_HASH_FUN */
int
hash_val(key, hashmask)
    const char *key;
    int hashmask;
{
  int hash = 0;
  const char *sp;

  /*
   * If the string pointer is null, return 0.  If not, add up the
   * numeric value of all the characters and return the sum,
   * modulo the size of the hash table.
   */

  if (!key || !*key)
    return 0;
  for (sp = key; *sp; sp++)
    hash = (hash << 5) + hash + *sp;
  return (hash & hashmask);
}
#endif				/* NEW_HASH_FUN */

/* ----------------------------------------------------------------------
 * hash_getmask: Get hash mask for mask-style hashing.
 */

int
hash_getmask(size)
    int *size;
{
  int tsize;

  if (!size || !*size)
    return 0;

  /* Get next power-of-two >= size, return power-1 as the mask
   * for ANDing
   */

  for (tsize = 1; tsize < *size; tsize = tsize << 1) ;
  *size = tsize;
  return tsize - 1;
}

void
hash_init(htab, size, data_size)
    HASHTAB *htab;
    int size;
    int data_size;
{
  int i;

  htab->mask = get_hashmask(&size);
  htab->hashsize = size;
  htab->entries = 0;
  htab->buckets = mush_malloc(size * sizeof(HASHENT *), "hash_buckets");
  for (i = 0; i < size; i++)
    htab->buckets[i] = NULL;

  htab->entry_size = data_size;
}

HASHENT *
hash_find(htab, key)
    HASHTAB *htab;
    const char *key;
{
  int hval, cmp;
  HASHENT *hptr;

  if (!htab->buckets)
    return NULL;

  hval = hash_val(key, htab->mask);
  for (hptr = htab->buckets[hval]; hptr != NULL; hptr = hptr->next) {
    cmp = strcmp(key, hptr->key);
    if (cmp == 0) {
      return hptr;
    } else if (cmp < 0)
      break;
  }
  return NULL;
}

void *
hash_value(entry)
    HASHENT *entry;
{
  if (entry)
    return entry->data;
  else
    return NULL;
}

char *
hash_key(entry)
    HASHENT *entry;
{
  if (entry)
    return entry->key;
  else
    return NULL;
}

void
hash_resize(htab, size)
    HASHTAB *htab;
    int size;
{
  int i;
  HASHENT **oldarr;
  HASHENT **newarr;
  HASHENT *hent, *nent, *curr, *old;
  int hval;
  int osize;
  int mask;

  /* We don't want hashes outside these limits */
  if ((size < (1 << 4)) || (size > (1 << 20)))
    return;

  /* Save the old data we need */
  osize = htab->hashsize;
  oldarr = htab->buckets;

  mask = htab->mask = get_hashmask(&size);

  if (size == htab->hashsize)
    return;

  htab->hashsize = size;
  newarr =
    (HASHENT **) mush_malloc(size * sizeof(struct hashentry *), "hash_buckets");
  htab->buckets = newarr;
  for (i = 0; i < size; i++)
    newarr[i] = NULL;

  for (i = 0; i < osize; i++) {
    hent = oldarr[i];
    while (hent) {
      nent = hent->next;
      hval = hash_val(hent->key, mask);
      for (curr = newarr[hval], old = NULL; curr; old = curr, curr = curr->next) {
	if (strcmp(curr->key, hent->key) > 0)
	  break;
      }
      if (old) {
	old->next = hent;
	hent->next = curr;
      } else {
	hent->next = newarr[hval];
	newarr[hval] = hent;
      }
      hent = nent;
    }
  }
  mush_free(oldarr, "hash_buckets");

  return;
}

HASHENT *
hash_new(htab, key)
    HASHTAB *htab;
    const char *key;
{
  int hval;
  Size_t keylen;
  HASHENT *hptr, *curr, *old;

  hptr = hash_find(htab, key);
  if (hptr)
    return hptr;

  if (htab->entries > (htab->hashsize * HTAB_UPSCALE))
    hash_resize(htab, htab->hashsize << 1);

  hval = hash_val(key, htab->mask);
  htab->entries++;
  keylen = strlen(key) + 1;
  hptr = (HASHENT *) mush_malloc(HASHENT_SIZE + keylen, "hash_entry");
  memcpy(hptr->key, key, keylen);
  hptr->data = NULL;

  if (!htab->buckets[hval] || strcmp(key, htab->buckets[hval]->key) < 0) {
    hptr->next = htab->buckets[hval];
    htab->buckets[hval] = hptr;
    return hptr;
  }

  /* Insert in sorted order. There's always at least one item in 
     the chain already at this point. */
  old = htab->buckets[hval];
  for (curr = old->next; curr; old = curr, curr = curr->next) {
    /* Comparison will never be 0 because hash_add checks to see
       if the entry is already present. */
    if (strcmp(key, curr->key) < 0) {	/* Insert before curr */
      old->next = hptr;
      hptr->next = curr;
      return hptr;
    }
  }

  /* If we get here, we reached the end of the chain */
  old->next = hptr;
  hptr->next = NULL;

  return hptr;
}

int
hash_add(htab, key, hashdata, extra_size)
    HASHTAB *htab;
    const char *key;
    void *hashdata;
    int extra_size __attribute__ ((__unused__));
{
  HASHENT *hptr;

  if (hash_find(htab, key) != NULL) {
    return -1;
  }

  hptr = hash_new(htab, key);

  if (!hptr)
    return -1;

  hptr->data = hashdata;
  /*      hptr->extra_size = extra_size; */
  return 0;
}

void
hash_delete(htab, entry)
    HASHTAB *htab;
    HASHENT *entry;
{
  int hval;
  HASHENT *hptr, *last;

  if (!entry)
    return;

  hval = hash_val(entry->key, htab->mask);
  last = NULL;
  for (hptr = htab->buckets[hval]; hptr; last = hptr, hptr = hptr->next) {
    if (entry == hptr) {
      if (last == NULL)
	htab->buckets[hval] = hptr->next;
      else
	last->next = hptr->next;
      mush_free(hptr, "hash_entry");
      htab->entries--;
      return;
    }
  }

  if (htab->entries < (htab->hashsize * HTAB_DOWNSCALE))
    hash_resize(htab, htab->hashsize >> 1);
}

void
hash_flush(htab, size)
    HASHTAB *htab;
    int size;
{
  HASHENT *hent, *thent;
  int i;

  if (htab->buckets) {
    for (i = 0; i < htab->hashsize; i++) {
      hent = htab->buckets[i];
      while (hent != NULL) {
	thent = hent;
	hent = hent->next;
	mush_free(thent, "hash_entry");
      }
      htab->buckets[i] = NULL;
    }
  }
  if (size == 0) {
    mush_free(htab->buckets, "hash_buckets");
    htab->buckets = NULL;
  } else if (size != htab->hashsize) {
    if (htab->buckets)
      mush_free(htab->buckets, "hash_buckets");
    hashinit(htab, size, htab->entry_size);
  } else {
    htab->entries = 0;
  }
}

void *
hash_firstentry(htab)
    HASHTAB *htab;
{
  int hval;

  for (hval = 0; hval < htab->hashsize; hval++)
    if (htab->buckets[hval]) {
      htab->last_hval = hval;
      htab->last_entry = htab->buckets[hval];
      return htab->buckets[hval]->data;
    }
  return NULL;
}

void *
hash_nextentry(htab)
    HASHTAB *htab;
{
  int hval;
  HASHENT *hptr;

  hval = htab->last_hval;
  hptr = htab->last_entry;
  if (hptr->next) {
    htab->last_entry = hptr->next;
    return hptr->next->data;
  }
  hval++;
  while (hval < htab->hashsize) {
    if (htab->buckets[hval]) {
      htab->last_hval = hval;
      htab->last_entry = htab->buckets[hval];
      return htab->buckets[hval]->data;
    }
    hval++;
  }
  return NULL;
}

void
hash_stats_header(player)
    dbref player;
{
  notify_format(player,
		"Table      Buckets Entries LChain  ECh  1Ch  2Ch  3Ch 4+Ch  AvgCh ~Memory");
}

void
hash_stats(player, htab, hname)
    dbref player;
    HASHTAB *htab;
    const char *hname;
{
  /* Report some statistics on the hash table */
  int longest = 0, n;
  int lengths[5];
  double chainlens = 0.0;
  double totchains = 0.0;
  unsigned int bytes = 0;

  if (!htab || !hname)
    return;

  for (n = 0; n < 5; n++)
    lengths[n] = 0;
  bytes += sizeof(HASHTAB);
  bytes += htab->entry_size * htab->entries;
  if (htab->buckets) {
    bytes += HASHENT_SIZE * htab->hashsize;
    for (n = 0; n < htab->hashsize; n++) {
      int chain = 0;
      HASHENT *b;
      if (htab->buckets[n]) {
	for (b = htab->buckets[n]; b; b = b->next) {
	  chain++;
	  bytes += strlen(b->key) + 1 /* + b->extra_size */ ;
	}
	if (chain > longest)
	  longest = chain;
      }
      lengths[(chain > 4) ? 4 : chain]++;
      chainlens += chain;
    }
  }
  for (n = 1; n < 5; n++)
    totchains += lengths[n];

  notify_format(player,
		"%-10s %7d %7d %6d %4d %4d %4d %4d %4d %6.3f %7u", hname,
		htab->hashsize, htab->entries, longest, lengths[0], lengths[1],
		lengths[2], lengths[3], lengths[4],
		totchains == 0.0 ? 0.0 : chainlens / totchains, bytes);
}
