/*-------------------------------------------------------------------*
 * malias.c - Global @mail aliases/lists
 *
 * This code implements an extension to extended @mail which allows
 * admin (and others who are so em@powered) to create mail aliases
 * for the MUSH. Optionally, any player can be allowed to.
 *
 * Aliases are used by @mail'ing to !<alias name>
 * Aliases have a name, a description, a list of members (dbrefs), an owner
 * a size (how many members), and two kinds of flags. 
 * nflags control who can use/see an alias name, and mflags 
 * control who can see the alias members. The choices
 * are everyone, alias members, owner, admin
 * 
 * Interface:
 * @malias[/list]
 * @malias/members !name
 * @malias[/create] !name=list-of-members
 * @malias/destroy !name
 * @malias/add !name=list-of-members
 * @malias/remove !name=list-of-members
 * @malias/desc !name=description
 * @malias/nameprivs !name=flags
 * @malias/listprivs !name=flags
 * @malias/stat
 * @malias/chown !name=owner    (Admin only)
 * @malias/nuke                 (Admin only)
 *-------------------------------------------------------------------*/

#define MA_INC 3
#include "config.h"
#include "copyrite.h"

#ifdef I_SYS_TIME
#include <sys/time.h>
#else
#include <time.h>
#endif
#include <ctype.h>
#ifdef I_SYS_TYPES
#include <sys/types.h>
#endif
#ifdef I_STRING
#include <string.h>
#else
#include <strings.h>
#endif

#include "conf.h"
#include "mushdb.h"
#include "dbdefs.h"
#include "externs.h"
#include "match.h"
#include "parse.h"
#include "malias.h"
#include "privtab.h"
#include "mymalloc.h"
#include "flags.h"
#include "pueblo.h"
#include "log.h"
#ifdef MEM_CHECK
#include "memcheck.h"
#endif
#include "confmagic.h"


#ifdef MAIL_ALIASES

int ma_size = 0;
int ma_top = 0;
struct mail_alias *malias;

static PRIV malias_priv_table[] = {
  {"Admin", 'A', ALIAS_ADMIN, ALIAS_ADMIN},
  {"Members", 'M', ALIAS_MEMBERS, ALIAS_MEMBERS},
  {"Owner", 'O', ALIAS_OWNER, ALIAS_OWNER},
  {NULL, '\0', 0, 0}
};

static const char *get_privs(int flags);
static const char *get_shortprivs(struct mail_alias *m);


/***********************************************************
***** User-commands *****
***********************************************************/


void
do_malias(player, arg1, arg2)
    dbref player;
    char *arg1;
    char *arg2;
{
  if (!arg1 || !*arg1) {
    if (arg2 && *arg2) {
      notify(player, T("MAIL: Invalid malias command."));
      return;
    }
    /* just the "@malias" command */
    do_malias_list(player);
    return;
  }
  if (arg2 && *arg2) {
    /* Creating malias */
    do_malias_create(player, arg1, arg2);
  } else {
    /* List specific alias - no arg2 */
    do_malias_members(player, arg1);
  }
}


void
do_malias_create(player, alias, tolist)
    dbref player;
    char *alias;
    char *tolist;
{
  char *head, *tail, spot;
  struct mail_alias *m;
  char *na;
  const char *buff, *good, *scan;
  int i = 0;
  dbref target;
  dbref alist[100];

  if (!IsPlayer(player)) {
    notify(player, T("MAIL: Only players may create mail aliases."));
    return;
  }
  if (!alias || !*alias || !tolist || !*tolist) {
    notify(player, T("MAIL: What alias do you want to create?."));
    return;
  }
  if (*alias != MALIAS_TOKEN) {
    notify_format(player,
		  T("MAIL: All Mail aliases must begin with '%c'."),
		  MALIAS_TOKEN);
    return;
  }
  good = "`$_-.'";
  /* Make sure that the name contains legal characters only */
  for (scan = alias + 1; scan && *scan; scan++) {
    if (isalpha((unsigned char) *scan) || isdigit((unsigned char) *scan))
      continue;
    if (!strchr(good, *scan)) {
      notify(player, T("MAIL: Invalid character in mail alias."));
      return;
    }
  }
  m = get_malias(GOD, alias);	/* GOD can see all aliases */
  if (m) {			/* Ensures no duplicates!  */
    notify_format(player, T("MAIL: Mail Alias '%s' already exists."), alias);
    return;
  }
  if (!ma_size) {
    ma_size = MA_INC;
    malias =
      (struct mail_alias *) mush_malloc(sizeof(struct mail_alias) *
					ma_size, "malias_list");
  } else if (ma_top >= ma_size) {
    ma_size += MA_INC;
    m =
      (struct mail_alias *) mush_malloc(sizeof(struct mail_alias) *
					(ma_size), "malias_list");
    memcpy(m, malias, sizeof(struct mail_alias) * ma_top);
    mush_free((Malloc_t) malias, "malias_list");
    malias = m;
  }
  i = 0;

  /*
   * Parse the player list
   */
  head = (char *) tolist;
  while (head && *head) {
    while (*head == ' ')
      head++;
    tail = head;
    while (*tail && (*tail != ' ')) {
      if (*tail == '"') {
	head++;
	tail++;
	while (*tail && (*tail != '"'))
	  tail++;
      }
      if (*tail)
	tail++;
    }
    tail--;
    if (*tail != '"')
      tail++;
    spot = *tail;
    *tail = '\0';
    /*
     * Now locate a target
     */
    if (!strcasecmp(head, "me"))
      target = player;
    else if (*head == '#') {
      target = atoi(head + 1);
    } else
      target = lookup_player(head);
    if (!(GoodObject(target)) || (!IsPlayer(target))) {
      notify_format(player, T("MAIL: No such player '%s'."), head);
    } else {
      buff = unparse_object(player, target);
      notify_format(player, T("MAIL: %s added to alias %s"), buff, alias);
      alist[i] = target;
      i++;
    }
    /*
     * Get the next recip
     */
    *tail = spot;
    head = tail;
    if (*head == '"')
      head++;
    if (i == 100)
      break;
  }

  if (head && *head) {
    notify(player, T("MAIL: Alias list is restricted to maximal 100 entries!"));
  }
  if (!i) {
    notify(player, T("MAIL: No valid recipients for alias-list!"));
    return;
  }
  m = &malias[ma_top];
  m->members = (dbref *) mush_malloc(sizeof(dbref) * i, "malias_members");
  memcpy(m->members, alist, sizeof(dbref) * i);

  na = alias + 1;
  m->size = i;
  m->owner = player;
  m->name = mush_strdup(na, "malias_name");
  m->desc = compress(na);
#ifdef MEM_CHECK
  add_check("malias_desc");
#endif
  m->nflags = ALIAS_OWNER | ALIAS_MEMBERS;
  m->mflags = ALIAS_OWNER;
  ma_top++;


  notify_format(player, T("MAIL: Alias set '%s' defined."), alias);
}



void
do_malias_list(player)
    dbref player;
{
  struct mail_alias *m;
  int i = 0;
  int notified = 0;

  for (i = 0; i < ma_top; i++) {
    m = &malias[i];
    if ((m->owner == player) || (m->nflags == 0) ||
	((m->nflags & ALIAS_ADMIN) && Hasprivs(player)) ||
	((m->nflags & ALIAS_MEMBERS) && ismember(m, player))) {
      if (!notified) {
	notify_format(player, "%-13s %-35s %s %-15s",
		      T("Name"), T("Alias Description"), T("Use See"),
		      T("Owner"));
	notified++;
      }
      notify_format(player,
		    "%c%-12.12s %-35.35s %s %-15.15s", MALIAS_TOKEN, m->name,
		    uncompress((unsigned char *) (m->desc)), get_shortprivs(m),
		    Name(m->owner));
    }
  }

  notify(player, T("*****  End of Mail Aliases *****"));
}

void
do_malias_members(player, alias)
    dbref player;
    char *alias;
{
  struct mail_alias *m;
  int i = 0;
  char buff[BUFFER_LEN];
  char *bp;

  m = get_malias(player, alias);

  if (!m) {
    notify_format(player, T("MAIL: Alias '%s' not found."), alias);
    return;
  }
  if ((m->owner == player) || (m->mflags == 0) ||
      (Hasprivs(player)) ||
      ((m->mflags & ALIAS_MEMBERS) && ismember(m, player))) {
    /* Dummy to avoid having to invert the "if" above ;-) */
  } else {
    notify(player, T("MAIL: Permission denied."));
    return;
  }
  bp = buff;
  safe_format(buff, &bp, T("MAIL: Alias %c%s: "), MALIAS_TOKEN, m->name);
  for (i = 0; i < m->size; i++) {
    safe_str(Name(m->members[i]), buff, &bp);
    safe_chr(' ', buff, &bp);
    /* Attention if player names may contain spaces!! */
  }
  *bp = '\0';
  notify(player, buff);
}



void
do_malias_desc(player, alias, desc)
    dbref player;
    char *alias;
    char *desc;
{
  struct mail_alias *m;

  if (!(m = get_malias(player, alias))) {
    notify_format(player, T("MAIL: Alias %s not found."), alias);
    return;
  } else if (Wizard(player) || (player == m->owner)) {
    if (m->desc)
      free(m->desc);		/* No need to update MEM_CHECK records here */
    m->desc = compress(desc);
    notify(player, T("MAIL: Description changed."));
  } else
    notify(player, T("MAIL: Permission denied."));
  return;
}


void
do_malias_chown(player, alias, owner)
    dbref player;
    char *alias;
    char *owner;
{
  struct mail_alias *m;
  dbref no = NOTHING;

  if (!(m = get_malias(player, alias))) {
    notify_format(player, T("MAIL: Alias %s not found."), alias);
    return;
  } else {
    if (!Wizard(player)) {
      notify(player, T("MAIL: You cannot do that!"));
      return;
    } else {
      if ((no = lookup_player(owner)) == NOTHING) {
	notify(player, T("MAIL: I cannot find that player."));
	return;
      }
      m->owner = no;
      notify(player, T("MAIL: Owner changed for alias."));
    }
  }
}



void
do_malias_rename(player, alias, newname)
    dbref player;
    char *alias;
    char *newname;
{
  struct mail_alias *m;

  if ((m = get_malias(player, alias)) == NULL) {
    notify(player, T("MAIL: I cannot find that alias!"));
    return;
  }
  if (*newname != MALIAS_TOKEN) {
    notify_format(player,
		  T("MAIL: Bad alias. Aliases must start with '%c'."),
		  MALIAS_TOKEN);
    return;
  }
  if (get_malias(GOD, newname) != NULL) {
    notify(player, T("MAIL: That name already exists!"));
    return;
  }
  if (!Wizard(player) && !(m->owner == player)) {
    notify(player, T("MAIL: Permission denied."));
    return;
  }

  free(m->name);		/* No need to update MEM_CHECK records here. */
  m->name = strdup(newname + 1);

  notify(player, T("MAIL: Mail Alias renamed."));
}



void
do_malias_destroy(player, alias)
    dbref player;
    char *alias;
{
  struct mail_alias *m;
  m = get_malias(player, alias);
  if (!m) {
    notify(player,
	   T
	   ("MAIL: Not a valid alias. Remember to prefix the alias name with *."));
    return;
  }
  if (Wizard(player) || (m->owner == player)) {
    notify(player, T("MAIL: Alias Destroyed."));
    if (m->members)
      mush_free((Malloc_t) m->members, "malias_members");
    if (m->name)
      mush_free(m->name, "malias_name");
    if (m->desc)
      mush_free(m->desc, "malias_desc");
    *m = malias[--ma_top];
  } else {
    notify(player, T("MAIL: Permission denied!"));
  }
}


void
do_malias_set(player, alias, tolist)
    dbref player;
    char *alias;
    char *tolist;
{
  struct mail_alias *m;
  int i = 0;
  char *head, *tail, spot;
  const char *buff;
  dbref alist[100];
  dbref target;

  m = get_malias(player, alias);
  if (!m) {
    notify_format(player,
		  T
		  ("MAIL: Not a valid alias. Remember to prefix the alias name with %c."),
		  MALIAS_TOKEN);
    return;
  }
  if (!tolist || !*tolist) {
    notify(player, T("MAIL: You must set the alias to a non-empty list."));
    return;
  }
  if (!(Wizard(player) || (m->owner == player))) {
    notify(player, T("MAIL: Permission denied!"));
    return;
  }

  /*
   * Parse the player list
   */
  head = (char *) tolist;
  while (head && *head) {
    while (*head == ' ')
      head++;
    tail = head;
    while (*tail && (*tail != ' ')) {
      if (*tail == '"') {
	head++;
	tail++;
	while (*tail && (*tail != '"'))
	  tail++;
      }
      if (*tail)
	tail++;
    }
    tail--;
    if (*tail != '"')
      tail++;
    spot = *tail;
    *tail = '\0';
    /*
     * Now locate a target
     */
    if (!strcasecmp(head, "me"))
      target = player;
    else if (*head == '#') {
      target = atoi(head + 1);
    } else
      target = lookup_player(head);
    if (!(GoodObject(target)) || (!IsPlayer(target))) {
      notify_format(player, T("MAIL: No such player '%s'."), head);
    } else {
      buff = unparse_object(player, target);
      notify_format(player, T("MAIL: %s added to alias %s"), buff, alias);
      alist[i] = target;
      i++;
    }
    /*
     * Get the next recip
     */
    *tail = spot;
    head = tail;
    if (*head == '"')
      head++;
    if (i == 100)
      break;
  }

  if (head && *head) {
    notify(player, T("MAIL: Alias list is restricted to maximal 100 entries!"));
  }
  if (!i) {
    notify(player, T("MAIL: No valid recipients for alias-list!"));
    return;
  }
  if (m->members)
    mush_free((Malloc_t) m->members, "malias_members");
  m->members = (dbref *) mush_malloc(sizeof(dbref) * i, "malias_members");
  memcpy(m->members, alist, sizeof(dbref) * i);
  m->size = i;
  notify(player, T("MAIL: Alias list set."));
}



void
do_malias_all(player)
    dbref player;
{
  struct mail_alias *m;
  int i;

  if (!Hasprivs(player)) {
    do_malias_list(player);
    return;
  }
  notify(player,
	 "Num   Name       Description                              Owner       Count");

  for (i = 0; i < ma_top; i++) {
    m = &malias[i];
    notify_format(player, "#%-4d %c%-10.10s %-40.40s %-11.11s (%3d)",
		  i, MALIAS_TOKEN, m->name,
		  uncompress((unsigned char *) m->desc),
		  Name(m->owner), m->size);
  }

  notify(player, T("***** End of Mail Aliases *****"));
}




void
do_malias_stats(player)
    dbref player;
{
  if (!Hasprivs(player))
    notify(player, T("MAIL: Permission denied."));
  else {
    notify_format(player,
		  T("MAIL: Number of mail aliases defined: %d"), ma_top);
    notify_format(player, T("MAIL: Allocated slots %d"), ma_size);
  }
}

void
do_malias_nuke(player)
    dbref player;
{
  struct mail_alias *m;
  int i;

  if (!God(player)) {
    notify(player, T("MAIL: Only god can do that!"));
    return;
  }
  if (ma_size) {		/* aliases defined ? */
    for (i = 0; i < ma_top; i++) {
      m = &malias[i];
      if (m->name)
	mush_free(m->name, "malias_name");
      if (m->desc)
	mush_free(m->desc, "malias_desc");
      if (m->members)
	mush_free((Malloc_t) m->members, "malias_members");
    }
    mush_free((Malloc_t) malias, "malias_list");
  }
  ma_size = ma_top = 0;
  notify(player, T("MAIL: All mail aliases destroyed!"));
}



void
do_malias_privs(player, alias, privs, type)
    dbref player;
    char *alias;
    char *privs;
    int type;			/* 0 = nprivs, 1 = mprivs */
{
  struct mail_alias *m;
  int *p;

  if (!(m = get_malias(player, alias))) {
    notify(player, T("MAIL: I cannot find that alias!"));
    return;
  }
  if (!Wizard(player) && (m->owner != player)) {
    notify(player, T("MAIL: Permission denied."));
    return;
  }
  p = type ? &m->mflags : &m->nflags;
  *p = string_to_privs(malias_priv_table, privs, 0);
  notify_format(player,
		T("MAIL: Permission to see/use alias '%s' changed to %s"),
		alias, privs_to_string(malias_priv_table, *p));
}


void
do_malias_mprivs(player, alias, privs)
    dbref player;
    char *alias;
    char *privs;
{
  const char TYPES[] = " everyone admin members owner";
  struct mail_alias *m;
  char *ch;

  if (!(m = get_malias(player, alias))) {
    notify(player, T("MAIL: I cannot find that alias!"));
    return;
  }
  if (!Wizard(player) && (m->owner != player)) {
    notify(player, T("MAIL: Permission denied."));
    return;
  }
  if (!strchr("=+-", *privs)) {
    notify(player,
	   T
	   ("MAIL: First character of permisisons must be one of '=', '+', '-',"));
    notify(player, T("      to set, add, remove permissions, respectively."));
    return;
  }
  ch = strstr(TYPES, tprintf(" %s", privs + 1));
  if (!ch) {
    notify_format(player, T("MAIL: Unknown permission type '%s'."), privs + 1);
    notify(player,
	   T
	   ("Must be one of 'everyone', 'members', 'admin', 'owner' (can be abbreviated)."));
    return;
  }
  if (*privs == '=') {		/* setting permissions */
    if (*ch == 'e')
      m->mflags = 0;
    else if (*ch == 'm')
      m->mflags = ALIAS_MEMBERS;
    else if (*ch == 'a')
      m->mflags = ALIAS_ADMIN;
    else if (*ch == 'o')
      m->mflags = ALIAS_OWNER;
  } else if ((*privs == '+') && m->mflags) {	/* adding permissions */
    if (*ch == 'm')
      m->mflags |= ALIAS_MEMBERS;
    else if (*ch == 'a')
      m->mflags |= ALIAS_ADMIN;
    else if (*ch == 'e') {	/* DUMMY */
    } else if (*ch == 'o') {	/* DUMMY */
    }
  } else if (*privs == '-') {	/* removing permissions */
    if (!m->mflags)
      m->mflags = ALIAS_ADMIN | ALIAS_MEMBERS;
    if (*ch == 'm')
      m->mflags &= ~ALIAS_MEMBERS;
    else if (*ch == 'a')
      m->mflags &= ~ALIAS_ADMIN;
    else if (*ch == 'e') {	/* DUMMY */
    } else if (*ch == 'o') {	/* DUMMY */
    }
    if (!m->mflags)
      m->mflags = ALIAS_OWNER;
  }
  notify_format(player,
		T("MAIL: Permission to list alias '%s' changed to %s"),
		alias, get_privs(m->mflags));
}




void
do_malias_add(player, alias, tolist)
    dbref player;
    char *alias;
    char *tolist;
{
  char *head, *tail, spot;
  struct mail_alias *m;
  const char *buff;
  int i = 0;
  dbref target;
  dbref alist[100];
  dbref *members;

  m = get_malias(player, alias);
  if (!m) {
    notify_format(player, T("MAIL: Mail Alias '%s' not found."), alias);
    return;
  }
  if (!Wizard(player) && (m->owner != player)) {
    notify(player, T("Permission denied."));
    return;
  }
  i = 0;

  /*
   * Parse the player list
   */
  head = (char *) tolist;
  while (head && *head) {
    while (*head == ' ')
      head++;
    tail = head;
    while (*tail && (*tail != ' ')) {
      if (*tail == '"') {
	head++;
	tail++;
	while (*tail && (*tail != '"'))
	  tail++;
      }
      if (*tail)
	tail++;
    }
    tail--;
    if (*tail != '"')
      tail++;
    spot = *tail;
    *tail = '\0';
    /*
     * Now locate a target
     */
    if (!strcasecmp(head, "me"))
      target = player;
    else if (*head == '#') {
      target = atoi(head + 1);
    } else
      target = lookup_player(head);
    if (!(GoodObject(target)) || (!IsPlayer(target))) {
      notify_format(player, T("MAIL: No such player '%s'."), head);
    } else {
      if (ismember(m, target)) {
	notify_format(player,
		      T("MAIL: player '%s' exists already in alias %s."),
		      head, alias);
      } else {
	buff = unparse_object(player, target);
	notify_format(player, T("MAIL: %s added to alias %s"), buff, alias);
	alist[i] = target;
	i++;
      }
    }
    /*
     * Get the next recip
     */
    *tail = spot;
    head = tail;
    if (*head == '"')
      head++;
    if (i == 100)
      break;
  }

  if (head && *head) {
    notify(player, T("MAIL: Alias list is restricted to maximal 100 entries!"));
  }
  if (!i) {
    notify(player, T("MAIL: No valid recipients for alias-list!"));
    return;
  }
  members =
    (dbref *) mush_malloc(sizeof(dbref) * (i + m->size), "malias_members");

  memcpy(members, m->members, sizeof(dbref) * m->size);
  memcpy(&members[m->size], alist, sizeof(dbref) * i);
  mush_free((Malloc_t) m->members, "malias_members");
  m->members = members;

  m->size += i;

  notify_format(player, T("MAIL: Alias set '%s' redefined."), alias);
}





void
do_malias_remove(player, alias, tolist)
    dbref player;
    char *alias;
    char *tolist;
{
  char *head, *tail, spot;
  struct mail_alias *m;
  const char *buff;
  int i = 0;
  dbref target;

  m = get_malias(player, alias);
  if (!m) {
    notify_format(player, T("MAIL: Mail Alias '%s' not found."), alias);
    return;
  }
  if (!Wizard(player) && (m->owner != player)) {
    notify(player, T("Permission denied."));
    return;
  }
  i = 0;

  /*
   * Parse the player list
   */
  head = (char *) tolist;
  while (head && *head) {
    while (*head == ' ')
      head++;
    tail = head;
    while (*tail && (*tail != ' ')) {
      if (*tail == '"') {
	head++;
	tail++;
	while (*tail && (*tail != '"'))
	  tail++;
      }
      if (*tail)
	tail++;
    }
    tail--;
    if (*tail != '"')
      tail++;
    spot = *tail;
    *tail = '\0';
    /*
     * Now locate a target
     */
    if (!strcasecmp(head, "me"))
      target = player;
    else if (*head == '#') {
      target = atoi(head + 1);
    } else
      target = lookup_player(head);
    if (!(GoodObject(target)) || (!IsPlayer(target))) {
      notify_format(player, T("MAIL: No such player '%s'."), head);
    } else {
      if (!(i = ismember(m, target))) {
	notify_format(player, T("MAIL: player '%s' is not in alias %s."),
		      head, alias);
      } else {
	buff = unparse_object(player, target);
	m->members[i - 1] = m->members[--m->size];
	notify_format(player, T("MAIL: %s removed from alias %s"), buff, alias);
      }
    }
    /*
     * Get the next recip
     */
    *tail = spot;
    head = tail;
    if (*head == '"')
      head++;
  }

  notify_format(player, T("MAIL: Alias set '%s' redefined."), alias);
}






/***********************************************************
***** "Utility" functions *****
***********************************************************/

static const char *
get_privs(flags)
    int flags;
{
  static char privs[20];
  privs[0] = '\0';
  if (!flags)
    strcpy(privs, "EVERYONE");
  else {
    if (flags & ALIAS_MEMBERS)
      strcpy(privs, "MEMBERS");
    if (flags & ALIAS_ADMIN) {
      if (*privs)
	strcat(privs, " | ");
      strcat(privs, "ADMIN");
    }
    if (!(*privs))
      strcpy(privs, "OWNER");
  }

  return privs;
}


static const char *
get_shortprivs(m)
    struct mail_alias *m;
{
  static char privs[10];
  strcpy(privs, "--  -- ");

  if (!m->nflags)
    privs[0] = 'E';
  else {
    if (m->nflags & ALIAS_MEMBERS)
      privs[0] = 'M';
    if (m->nflags & ALIAS_ADMIN)
      privs[1] = 'A';
    if (!strncmp(privs, "--", 2))
      privs[1] = 'O';
  }

  if (!m->mflags)
    privs[4] = 'E';
  else {
    if (m->mflags & ALIAS_MEMBERS)
      privs[4] = 'M';
    if (m->mflags & ALIAS_ADMIN)
      privs[5] = 'A';
    if (!strncmp(privs + 4, "--", 2))
      privs[5] = 'O';
  }

  return privs;
}


int
ismember(m, player)
    struct mail_alias *m;
    dbref player;
{
  int i;
  for (i = 0; i < m->size; i++) {
    if (player == m->members[i])
      return (i + 1);		/* To avoid entry "0" */
  }
  return 0;
}

/* Remove a destroyed player from all maliases */
void
malias_cleanup(player)
    dbref player;
{
  struct mail_alias *m;
  int n, i = 0;

  for (n = 0; n < ma_top; n++) {
    m = &malias[n];
    if ((i = ismember(m, player)) != 0) {
      do_rawlog(LT_ERR, "Removing #%d from malias %s", player, m->name);
      m->members[i - 1] = m->members[--m->size];
    }
  }
}

struct mail_alias *
get_malias(player, alias)
    dbref player;
    char *alias;
{
  const char *mal;
  struct mail_alias *m;
  int i = 0;
  int x = 0;

  if ((*alias == '#') && Hasprivs(player)) {
    x = atoi(alias + 1);
    if (x < 0 || x > (ma_top - 1))
      return NULL;
    else
      return &malias[x];
  } else {
    if (*alias != MALIAS_TOKEN)
      return NULL;

    mal = alias + 1;

    for (i = 0; i < ma_top; i++) {
      m = &malias[i];
      if ((m->owner == player) || (m->nflags == 0) ||
	  /* ((m->nflags & ALIAS_ADMIN) && Hasprivs(player)) || */
	  Hasprivs(player) ||
	  ((m->nflags & ALIAS_MEMBERS) && ismember(m, player))) {

#ifdef macintosh
	if (!strcasecmp(mal, (char *) (m->name)))
#else
	if (!strcasecmp(mal, m->name))
#endif
	  return m;
      }
    }
  }
  return NULL;
}








/***********************************************************
***** Loading and saving of mail-aliases *****
***********************************************************/


void
load_malias(fp)
    FILE *fp;
{
  int i, j;
  char buffer[BUFFER_LEN];
  struct mail_alias *m;
  char *s;

  ma_top = getref(fp);

  ma_size = ma_top;

  if (ma_top > 0)
    malias =
      (struct mail_alias *) mush_malloc(sizeof(struct mail_alias) *
					ma_size, "malias_list");
  else
    malias = NULL;

  for (i = 0; i < ma_top; i++) {
    m = &malias[i];

    m->owner = getref(fp);
    m->name = mush_strdup(getstring_noalloc(fp), "malias_name");
    m->desc = compress(getstring_noalloc(fp));
#ifdef MEM_CHECK
    add_check("malias_desc");
#endif

    m->nflags = getref(fp);
    m->mflags = getref(fp);
    m->size = getref(fp);

    if (m->size > 0) {
      m->members =
	(dbref *) mush_malloc(m->size * sizeof(dbref), "malias_members");
      for (j = 0; j < m->size; j++) {
	m->members[j] = getref(fp);
      }
    } else {
      m->members = NULL;
    }
  }
  s = fgets(buffer, sizeof(buffer), fp);

  if (!s || strcmp(buffer, "\"*** End of MALIAS ***\"\n") != 0) {
    do_rawlog(LT_ERR, T("MAIL: Error reading MALIAS list"));
  }
}


void
save_malias(fp)
    FILE *fp;
{
  int i, j;
  struct mail_alias *m;

  putref(fp, ma_top);

  for (i = 0; i < ma_top; i++) {
    m = &malias[i];
    putref(fp, m->owner);
    putstring(fp, (char *) (m->name));
    putstring(fp, uncompress(m->desc));

    putref(fp, m->nflags);
    putref(fp, m->mflags);
    putref(fp, m->size);

    for (j = 0; j < m->size; j++)
      putref(fp, m->members[j]);
  }
  putstring(fp, "*** End of MALIAS ***");
  /* putstring() of fputs() ???? 
   * Stupid thing to do. Someone sets a malias desc of "*** End of MALIAS ***"
   * and then that maildb is read into a mush without MAIL_ALIAS, and bad 
   * stuff happens. It's rare, but it might happen. Will fix later. */
}



#else				/* MAIL_ALIASES */

void
load_malias(fp)			/* Skip mail aliases,           */
    FILE *fp;			/* when they shall not be used, */
{
  char buffer[BUFFER_LEN];
  char *s;

  s = fgets(buffer, sizeof(buffer), fp);

  while (s && strcmp(buffer, "\"*** End of MALIAS ***\"\n") != 0) {
    s = fgets(buffer, sizeof(buffer), fp);
  }
}

#endif				/* MAIL_ALIASES */
