/* match.c */
/* Routines for parsing arguments */

/***********************************************************************
 * These are the PennMUSH name-matching routines, rewritten by
 * Javelin to be fully re-entrant instead of using global variables,
 * which fixes some bugs. [pl12]
 *
 * Calling these routines no longer requires using init_match,
 * a bunch of match_<foo> calls, and then getting match_result().
 * Instead, call one of these functions:
 *  match_result(who,name,type,flags) - return match, AMBIGUOUS, or NOTHING
 *  noisy_match_result(who,name,type,flags) - return match or NOTHING,
 *      and notify player on failures
 *  last_match_result(who,name,type,flags) - return match or NOTHING,
 *      and return the last match found in ambiguous situations
 *
 * who = dbref of player to match for
 * name = string to match on
 * type = preferred type of match (TYPE_THING, etc.) or NOTYPE
 * flags = a set of bits indicating what kind of matching to do
 *
 * flags are defined in match.h, but here they are for reference:
 * MAT_CHECK_KEYS       - check locks when matching
 * MAT_GLOBAL           - match in master room
 * MAT_REMOTES          - match things not nearby
 * MAT_NEAR             - match things nearby
 * MAT_CONTROL          - do a control check after matching
 * MAT_ME               - match "me"
 * MAT_HERE             - match "here"
 * MAT_ABSOLUTE         - match "#<dbref>"
 * MAT_PLAYER           - match a player's name
 * MAT_NEIGHBOR         - match something in the same room
 * MAT_POSSESSION       - match something I'm carrying
 * MAT_EXIT             - match an exit
 * MAT_CARRIED_EXIT     - match a carried exit (rare)
 * MAT_CONTAINER        - match a container I'm in
 * MAT_REMOTE_CONTENTS  - match the contents of a remote location
 * MAT_ENGLISH          - match natural english 'my 2nd flower'
 * MAT_EVERYTHING       - me,here,absolute,player,neighbor,possession,exit
 * MAT_NEARBY           - everything near
 * MAT_OBJECTS          - me,absolute,player,neigbor,possession
 * MAT_NEAR_THINGS      - objects near
 * MAT_REMOTE           - absolute,player,remote_contents,exit,remotes
 * MAT_LIMITED          - absolute,player,neighbor
 */

#include "copyrite.h"
#include "config.h"
#include <ctype.h>
#ifdef I_STRING
#include <string.h>
#else
#include <strings.h>
#endif
#ifdef I_STDLIB
#include <stdlib.h>
#endif
#include "conf.h"
#include "mushdb.h"
#include "externs.h"
#include "case.h"
#include "match.h"
#include "parse.h"
#include "flags.h"
#include "dbdefs.h"
#include "confmagic.h"

static dbref match_result_internal
  (dbref who, const char *name, int type, long flags);
static dbref match_me(const dbref who, const char *name);
static dbref match_here(const dbref who, const char *name);
static dbref match_absolute(const char *name);
static dbref match_player(const dbref matcher, const char *match_name);
static dbref match_list(const dbref match_who, const char *match_name,
			const int type, const long int flags, dbref first,
			dbref *last_match, int *match_count);
static dbref match_neighbor(const dbref who, const char *name,
			    const int type, const long int flags,
			    dbref *last_match, int *match_count);
static dbref match_possession(const dbref who, const char *name,
			      const int type, const long int flags,
			      dbref *last_match, int *match_count);
static dbref match_exit(const dbref who, const char *name,
			const int type, const long int flags, int *match_count);
static dbref match_exit_internal(const dbref match_who,
				 const char *match_name, const int type,
				 const long int flags, const dbref loc,
				 int *match_count);
static dbref match_remote_contents(const dbref who, const char *name,
				   const int type, const long int flags,
				   dbref *last_match, int *match_count);
static dbref match_container(const dbref who, const char *name,
			     const int type, const long int flags,
			     dbref *last_match, int *match_count);
static dbref match_english(const dbref who, const char *name,
			   const long int flags);
static dbref choose_thing(const dbref match_who, const int preferred_type,
			  long int flags, dbref thing1, dbref thing2,
			  int *match_count);
extern int check_alias(const char *command, const char *list);	/* game.c */


/* A wrapper for returning a match, AMBIGUOUS, or NOTHING
 */
dbref
match_result(who, name, type, flags)
    const dbref who;
    const char *name;
    const int type;
    const long int flags;
{
  return match_result_internal(who, name, type, flags);
}

/* A wrapper for returning a match or NOTHING
 * Ambiguous matches return NOTHING
 * It will also notify the player of non-matches or ambiguous matches
 */
dbref
noisy_match_result(who, name, type, flags)
    const dbref who;
    const char *name;
    const int type;
    const long int flags;
{
  return match_result_internal(who, name, type, flags | MAT_NOISY);
}

/* A wrapper for returning a match or NOTHING
 * Ambiguous matches return the last matched thing
 */
dbref
last_match_result(who, name, type, flags)
    const dbref who;
    const char *name;
    const int type;
    const long int flags;
{
  return match_result_internal(who, name, type, flags | MAT_LAST);
}

/* Wrapper for a noisy match with control checks */
dbref
match_controlled(player, name)
    dbref player;
    const char *name;
{
  dbref match;
  match = noisy_match_result(player, name, NOTYPE, MAT_EVERYTHING);
  if (GoodObject(match) && !controls(player, match)) {
    notify(player, T("Permission denied."));
    return NOTHING;
  } else {
    return match;
  }
}

/* The real work. */
static dbref
match_result_internal(dbref who, const char *name, int type, long flags)
{
  dbref match = NOTHING, last_match = NOTHING, exact_match = NOTHING;
  int exact_match_count = 0;
  int match_count = 0;
  if (flags & MAT_ME) {
    match = match_me(who, name);
    if (GoodObject(match))
      return match;
  }
  if (flags & MAT_HERE) {
    match = match_here(who, name);
    if (GoodObject(match))
      return match;
  }
  if (!(flags & MAT_NEAR) || Long_Fingers(who)) {
    if (flags & MAT_ABSOLUTE) {
      match = match_absolute(name);
      if (GoodObject(match)) {
	if (flags & MAT_CONTROL) {
	  /* Check for control */
	  if (controls(who, match) || nearby(who, match))
	    return match;
	} else {
	  return match;
	}
      }
    }
    if (flags & MAT_PLAYER) {
      match = match_player(who, name);
      if (GoodObject(match))
	return match;
    }
  } else {
    /* We're doing a nearby match and the player doesn't have
     * long_fingers, so it's a controlled absolute
     */
    match = match_absolute(name);
    if (GoodObject(match) && (controls(who, match) || nearby(who, match)))
      return match;
  }

  /* These return a match if the match is exact, and otherwise
   * store last thing matched in last_match and the number of matches
   * in match_count. Remote_contents and Neighbor are exclusive.
   * We need to deal with the possibility of having multiple exact
   * matches, multiple partial matches, or 1 exact + 1 or more partial
   * matches. 
   */
  if (DO_GLOBALS && (flags & MAT_REMOTE_CONTENTS)) {
    match =
      match_remote_contents(who, name, type, flags, &last_match, &match_count);
    if (GoodObject(match)) {
      exact_match_count += match_count;
      exact_match = match;
    }
  }
  if (flags & MAT_NEIGHBOR) {
    match = match_neighbor(who, name, type, flags, &last_match, &match_count);
    if (GoodObject(match)) {
      exact_match_count += match_count;
      exact_match = match;
    }
  }
  if (flags & MAT_POSSESSION) {
    match = match_possession(who, name, type, flags, &last_match, &match_count);
    if (GoodObject(match)) {
      exact_match_count += match_count;
      exact_match = match;
    }
  }
  if (flags & MAT_EXIT) {
    match = match_exit(who, name, type, flags, &match_count);
    if (GoodObject(match)) {
      exact_match_count += match_count;
      exact_match = match;
    }
  }
  if (flags & MAT_CONTAINER) {
    match = match_container(who, name, type, flags, &last_match, &match_count);
    if (GoodObject(match)) {
      exact_match_count += match_count;
      exact_match = match;
    }
  }
  if (flags & MAT_CARRIED_EXIT) {
    match = match_exit_internal(who, name, type, flags, who, &match_count);
    if (GoodObject(match)) {
      exact_match_count += match_count;
      exact_match = match;
    }
  }
  if ((flags & MAT_ENGLISH) && !GoodObject(exact_match)
      && !((flags & MAT_LAST) && GoodObject(last_match))) {
    match = match_english(who, name, flags);
    if (GoodObject(match)) {
      exact_match_count++;
      exact_match = match;
    }
  }

  /* Set up the default match_result behavior */
  if (GoodObject(exact_match)) {
    if (exact_match_count == 1)
      match = exact_match;
    else if (flags & MAT_LAST)
      match = exact_match;
    else
      match = AMBIGUOUS;
  } else {
    if (match_count == 0)
      match = NOTHING;
    else if (match_count == 1)
      match = last_match;
    else if (flags & MAT_LAST)
      match = last_match;
    else
      match = AMBIGUOUS;
  }

  /* Handle noisy_match_result */
  if (flags & MAT_NOISY) {
    switch (match) {
    case NOTHING:
      notify(who, T("I can't see that here."));
      return NOTHING;
    case AMBIGUOUS:
      notify(who, T("I don't know which one you mean!"));
      return NOTHING;
    default:
      return match;
    }
  }
  return match;
}

static dbref
match_me(who, name)
    const dbref who;
    const char *name;
{
  if (!strcasecmp(name, "me"))
    return who;
  return NOTHING;
}

static dbref
match_here(who, name)
    const dbref who;
    const char *name;
{
  if (!strcasecmp(name, "here") && GoodObject(Location(who)))
    return Location(who);
  return NOTHING;
}

static dbref
match_absolute(name)
    const char *name;
{
  return parse_dbref(name);
}

static dbref
match_player(matcher, match_name)
    const dbref matcher;
    const char *match_name;
{
  dbref match;
  const char *p;


  if (*match_name == LOOKUP_TOKEN) {
    for (p = match_name + 1; isspace((unsigned char) *p); p++) ;
    if ((match = lookup_player(p)) != NOTHING) {
      return match;
    } else {
      /* try a partial match on a connected player, 2.0 style */
      /* Can return a match, NOTHING, or AMBIGUOUS */
      return visible_short_page(matcher, p);
    }
  }
  return NOTHING;
}

/* This code should return a matched dbref only on an exact match.
 * Otherwise, it puts the best partial match in last_match
 */
static dbref
match_list(match_who, match_name, type, flags, first, last_match, match_count)
    const dbref match_who;
    const char *match_name;
    const int type;
    const long int flags;
    dbref first;
    dbref *last_match;
    int *match_count;
{
  dbref absolute;
  dbref the_match = NOTHING;
  dbref alias_match;
  absolute = match_absolute(match_name);
  if (!controls(match_who, absolute))
    absolute = NOTHING;

  alias_match = lookup_player(match_name);
  DOLIST(first, first) {
    if (first == absolute) {
      (*match_count)++;
      return first;
    } else if (!strcasecmp(Name(first), match_name) ||
	       (GoodObject(alias_match) && (alias_match == first))) {
      if (GoodObject(the_match)) {
	/* if there are multiple exact matches, don't match any but 
	 * make sure match_count is high enough that we report ambiguity
	 */
	*last_match =
	  choose_thing(match_who, type, flags, the_match, first, match_count);
	return NOTHING;
      } else {
	the_match = first;
	(*match_count)++;
      }
    } else if (string_match(Name(first), match_name)) {
      *last_match =
	choose_thing(match_who, type, flags, *last_match, first, match_count);
    }
  }
  return the_match;
}

static dbref
match_neighbor(who, name, type, flags, last_match, match_count)
    const dbref who;
    const char *name;
    const int type;
    const long int flags;
    dbref *last_match;
    int *match_count;
{
  dbref loc;
  loc = Location(who);
  if (GoodObject(loc)) {
    return match_list(who, name, type, flags, Contents(loc), last_match,
		      match_count);
  }
  return NOTHING;
}

static dbref
match_possession(who, name, type, flags, last_match, match_count)
    const dbref who;
    const char *name;
    const int type;
    const long flags;
    dbref *last_match;
    int *match_count;
{
  return match_list(who, name, type, flags, Contents(who), last_match,
		    match_count);
}

static dbref
match_exit(who, name, type, flags, match_count)
    const dbref who;
    const char *name;
    const int type;
    const long flags;
    int *match_count;
{
  dbref loc;
  loc = (IsRoom(who)) ? who : Location(who);
  if (DO_GLOBALS) {
    if (flags & MAT_REMOTES) {
      if (GoodObject(loc))
	return match_exit_internal(who, name, type, flags, Zone(loc),
				   match_count);
      else
	return NOTHING;
    } else if (flags & MAT_GLOBAL)
      return match_exit_internal(who, name, type, flags, MASTER_ROOM,
				 match_count);
  }
  return match_exit_internal(who, name, type, flags, loc, match_count);
}

static dbref
match_exit_internal(match_who, match_name, type, flags, loc, match_count)
    const dbref match_who;
    const char *match_name;
    const int type;
    const long int flags;
    const dbref loc;
    int *match_count;
{
  dbref exit_tmp;
  dbref absolute;
  dbref the_match = NOTHING;

  if (!GoodObject(loc) || !match_name || !*match_name)
    return NOTHING;
  if (!IsRoom(loc))
    return NOTHING;
  absolute = match_absolute(match_name);
  DOLIST(exit_tmp, Exits(loc)) {
    if (exit_tmp == absolute) {
      the_match = exit_tmp;
      if (match_count)
	(*match_count)++;
      break;
    } else if (check_alias(match_name, Name(exit_tmp)))
      the_match =
	choose_thing(match_who, type, flags, the_match, exit_tmp, match_count);
  }
  return the_match;
}


static dbref
match_remote_contents(who, name, type, flags, last_match, match_count)
    const dbref who;
    const char *name;
    const int type;
    const long int flags;
    dbref *last_match;
    int *match_count;
{
  if (GoodObject(who))
    return match_list(who, name, type, flags, Contents(who), last_match,
		      match_count);
  return NOTHING;
}

static dbref
match_container(who, name, type, flags, last_match, match_count)
    const dbref who;
    const char *name;
    const int type;
    const long int flags;
    dbref *last_match;
    int *match_count;
{
  dbref loc;
  loc = Location(who);
  if (GoodObject(loc))
    return match_list(who, name, type, flags, loc, last_match, match_count);
  return NOTHING;
}

static dbref
match_english(who, name, flags)
    const dbref who;
    const char *name;
    const long int flags;
{
  int do_loc;
  int do_cont;
  int do_exits, doing_exits;
  int num, matchnum;
  char *dupname, *p;
  char *mname;
  char *e;
  dbref item;

  if (!Mobile(who))
    return NOTHING;

  p = dupname = mush_strdup(name, "string");
  if (flags & MAT_NEIGHBOR)
    do_loc = 1;
  else
    do_loc = 0;
  if (flags & MAT_POSSESSION)
    do_cont = 1;
  else
    do_cont = 0;
  if (flags & MAT_EXIT)
    do_exits = 1;
  else
    do_exits = 0;

  if (do_loc && (!strncasecmp(dupname, "this here ", 10))) {
    dupname += 10;
    do_cont = do_exits = 0;
  } else if (do_loc && (!strncasecmp(dupname, "here ", 5)
			|| !strncasecmp(dupname, "this ", 5))) {
    dupname += 5;
    do_cont = do_exits = 0;
  } else if (do_cont && (!strncasecmp(dupname, "my ", 3)
			 || !strncasecmp(dupname, "me ", 3))) {
    dupname += 3;
    do_loc = do_exits = 0;
  } else if (do_exits && (!strncasecmp(dupname, "toward ", 7))) {
    dupname += 7;
    do_loc = do_cont = 0;
  }

  while (*dupname == ' ')
    dupname++;

  if (!*dupname) {
    mush_free(p, "string");
    return NOTHING;
  }

  mname = strchr(dupname, ' ');
  if (mname) {
    *mname = '\0';
    num = strtoul(dupname, &e, 10);
    if (e && *e) {
      if (num < 1) {
	num = -1;
      } else if ((num > 10) && (num < 14)) {
	if (strcasecmp(e, "th"))
	  num = -1;
      } else if ((num % 10) == 1) {
	if (strcasecmp(e, "st"))
	  num = -1;
      } else if ((num % 10) == 2) {
	if (strcasecmp(e, "nd"))
	  num = -1;
      } else if ((num % 10) == 3) {
	if (strcasecmp(e, "rd"))
	  num = -1;
      } else if (strcasecmp(e, "th")) {
	num = -1;
      }
    }

    if (num != -1) {
      dupname = mname + 1;
      while (*dupname == ' ')
	dupname++;
    } else {
      num = 1;
      *mname = ' ';
    }
  } else {
    num = 1;
  }

  doing_exits = 0;
  while (do_cont || do_loc || do_exits) {
    if (do_cont) {
      item = Contents(who);
      do_cont = 0;
    } else if (do_loc) {
      item = Contents(Location(who));
      do_loc = 0;
    } else {
      item = Exits(Location(who));
      doing_exits = 1;
      do_exits = 0;
    }
    matchnum = 0;
    DOLIST(item, item) {
      if (doing_exits ? (check_alias(dupname, Name(item)))
	  : (!strcasecmp(Name(item), dupname)
	     || string_match(Name(item), dupname))) {
	matchnum++;
	if (matchnum == num) {
	  mush_free(p, "string");
	  return item;
	}
      }
    }
  }
  mush_free(p, "string");
  return NOTHING;
}


static dbref
choose_thing(match_who, preferred_type, flags, thing1, thing2, match_count)
    const dbref match_who;
    const int preferred_type;
    long int flags;
    dbref thing1;
    dbref thing2;
    int *match_count;
{
  int has1;
  int has2;
  if (match_count)
    (*match_count)++;
  if (thing1 == NOTHING) {
    return thing2;
  } else if (thing2 == NOTHING) {
    return thing1;
  }
  if (preferred_type != NOTYPE) {
    if (Typeof(thing1) == preferred_type) {
      if (Typeof(thing2) != preferred_type) {
	return thing1;
      }
    } else if (Typeof(thing2) == preferred_type) {
      return thing2;
    }
  }
  if (flags & MAT_CHECK_KEYS) {
    has1 = could_doit(match_who, thing1);
    has2 = could_doit(match_who, thing2);

    if (has1 && !has2) {
      return thing1;
    } else if (has2 && !has1) {
      return thing2;
    }
    /* else fall through */
  }
  if (match_count)
    (*match_count)++;
  return (thing1 > thing2 ? thing1 : thing2);
}
