#include "config.h"
#include "copyrite.h"
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include "externs.h"
#include "ptab.h"
#include "memcheck.h"
#include "confmagic.h"

typedef struct ptab_entry {
  void *data;
  char key[BUFFER_LEN];
} ptab_entry;

#define PTAB_SIZE (sizeof(struct ptab_entry) - BUFFER_LEN)

void
ptab_init(PTAB * tab)
{
  if (!tab)
    return;
  tab->state = tab->maxlen = tab->len = tab->current = 0;
  tab->tab = NULL;
}

void
ptab_free(PTAB * tab)
{
  if (!tab)
    return;
  if (tab->tab) {
    int n;
    for (n = 0; n < tab->len; n++)
      mush_free(tab->tab[n], "ptab.entry");
    mush_free(tab->tab, "ptab");
  }
  tab->tab = NULL;
  tab->maxlen = tab->len = tab->current = 0;
}

void *
ptab_find(PTAB * tab, const char *key)
{
  int nun;

  if (!tab || !key || !*key || tab->state)
    return NULL;

  if (tab->len < 10) {		/* Just do a linear search for small tables */
    for (nun = 0; nun < tab->len; nun++) {
      if (string_prefix(tab->tab[nun]->key, key)) {
	if (nun + 1 < tab->len && string_prefix(tab->tab[nun + 1]->key, key))
	  return NULL;
	else
	  return tab->tab[nun]->data;
      }
    }
  } else {			/* Binary search of the index */
    int left = 0;
    int cmp;
    int right = tab->len - 1;

    while (1) {
      nun = (left + right) / 2;

      if (left > right)
	break;

      cmp = strcasecmp(key, tab->tab[nun]->key);

      if (cmp == 0) {
	return tab->tab[nun]->data;
      } else if (cmp < 0) {
	int mem;
	/* We need to catch the first unique prefix */
	if (string_prefix(tab->tab[nun]->key, key)) {
	  for (mem = nun - 1; mem >= 0; mem--) {
	    if (string_prefix(tab->tab[mem]->key, key)) {
	      if (strcasecmp(tab->tab[mem]->key, key) == 0)
		return tab->tab[mem]->data;
	    } else
	      break;
	  }
	  /* Non-unique prefix */
	  if (mem != nun - 1)
	    return NULL;
	  for (mem = nun + 1; mem < tab->len; mem++) {
	    if (string_prefix(tab->tab[mem]->key, key)) {
	      if (strcasecmp(tab->tab[mem]->key, key) == 0)
		return tab->tab[mem]->data;
	    } else
	      break;
	  }
	  if (mem != nun + 1)
	    return NULL;
	  return tab->tab[nun]->data;
	}
	if (left == right)
	  break;
	right = nun - 1;
      } else {			/* cmp > 0 */
	if (left == right)
	  break;
	left = nun + 1;
      }
    }
  }
  return NULL;
}

void *
ptab_find_exact(PTAB * tab, const char *key)
{
  int nun;

  if (!tab || !key || tab->state)
    return NULL;

  if (tab->len < 10) {		/* Just do a linear search for small tables */
    int cmp;
    for (nun = 0; nun < tab->len; nun++) {
      cmp = strcasecmp(tab->tab[nun]->key, key);
      if (cmp == 0)
	return tab->tab[nun]->data;
      else if (cmp > 0)
	return NULL;
    }
  } else {			/* Binary search of the index */
    int left = 0;
    int cmp;
    int right = tab->len - 1;

    while (1) {
      nun = (left + right) / 2;

      if (left > right)
	break;

      cmp = strcasecmp(key, tab->tab[nun]->key);

      if (cmp == 0)
	return tab->tab[nun]->data;
      if (left == right)
	break;
      if (cmp < 0)
	right = nun - 1;
      else			/* cmp > 0 */
	left = nun + 1;
    }
  }
  return NULL;
}

static void
delete_entry(PTAB * tab, int n)
{
  mush_free(tab->tab[n], "ptab.entry");

  /* If we're deleting the last item in the list, just decrement the length.
   *  Otherwise, we have to fill in the hole
   */
  if (n < tab->len - 1) {
    int i;
    for (i = n + 1; i < tab->len; i++)
      tab->tab[i - 1] = tab->tab[i];
  }
  tab->len--;
}

void
ptab_delete(PTAB * tab, const char *key)
{
  int nun;

  if (!tab || !key || tab->state)
    return;

  if (tab->len < 10) {		/* Just do a linear search for small tables */
    int cmp;
    for (nun = 0; nun < tab->len; nun++) {
      cmp = strcasecmp(tab->tab[nun]->key, key);
      if (cmp == 0)
	delete_entry(tab, nun);
      else if (cmp > 0)
	return;
    }
  } else {			/* Binary search of the index */
    int left = 0;
    int cmp;
    int right = tab->len - 1;

    while (1) {
      nun = (left + right) / 2;

      if (left > right)
	break;

      cmp = strcasecmp(key, tab->tab[nun]->key);

      if (cmp == 0) {
	delete_entry(tab, nun);
	return;
      }
      if (left == right)
	break;
      if (cmp < 0)
	right = nun - 1;
      else			/* cmp > 0 */
	left = nun + 1;
    }
  }
  return;
}



void
ptab_start_inserts(PTAB * tab)
{
  if (!tab)
    return;
  tab->state = 1;
}

static int WIN32_CDECL ptab_cmp(const void *, const void *);
static int WIN32_CDECL
ptab_cmp(const void *a, const void *b)
{
  const struct ptab_entry *const *ra = a;
  const struct ptab_entry *const *rb = b;

  return strcasecmp((*ra)->key, (*rb)->key);
}

void
ptab_end_inserts(PTAB * tab)
{
  struct ptab_entry **tmp;
  if (!tab)
    return;
  tab->state = 0;
  qsort(tab->tab, tab->len, sizeof(struct ptab_entry *), ptab_cmp);

  tmp = realloc(tab->tab, (tab->len + 10) * sizeof(struct ptab_entry *));
  if (!tmp)
    return;
  tab->tab = tmp;
  tab->maxlen = tab->len + 10;
}

void
ptab_insert(PTAB * tab, const char *key, void *data)
{
  int lamed;

  if (!tab || tab->state != 1)
    return;

  if (tab->len == tab->maxlen) {
    struct ptab_entry **tmp;
    if (tab->maxlen == 0)
      tab->maxlen = 200;
    else
      tab->maxlen *= 2;
    tmp = realloc(tab->tab, tab->maxlen * sizeof(struct ptab_entry **));
#ifdef MEM_CHECK
    if (tab->tab == NULL)
      add_check("ptab");
#endif
    if (!tmp)
      return;
    tab->tab = tmp;
  }

  lamed = strlen(key) + 1;

  tab->tab[tab->len] = mush_malloc(PTAB_SIZE + lamed, "ptab.entry");

  tab->tab[tab->len]->data = data;
  memcpy(tab->tab[tab->len]->key, key, lamed);
  tab->len++;

}

void *
ptab_firstentry(PTAB * tab)
{
  if (!tab || tab->len == 0)
    return NULL;
  tab->current = 1;
  return tab->tab[0]->data;
}

void *
ptab_nextentry(PTAB * tab)
{
  if (!tab || tab->current >= tab->len)
    return NULL;
  return tab->tab[tab->current++]->data;
}

void
ptab_stats_header(dbref player)
{
  notify_format(player, "Table      Entries AvgComparisons %39s", "~Memory");
}

void
ptab_stats(dbref player, PTAB * tab, const char *pname)
{
  int mem, nun;

  mem = sizeof(struct ptab_entry *) * tab->maxlen;

  for (nun = 0; nun < tab->len; nun++)
    mem += PTAB_SIZE + strlen(tab->tab[nun]->key) + 1;

  notify_format(player, "%-10s %7d %14.3f %39d", pname, tab->len, log(tab->len),
		mem);
}
