/* Signal handling routines for Penn. */

#include "config.h"
#include "externs.h"
#include <signal.h>

#include "confmagic.h"

#ifndef HAS_SIGPROCMASK
static Sigfunc saved_handlers[NSIG];
#endif

/* We're going to rewrite the signal() function in terms of
 * sigaction, where available, to ensure consistent semantics.
 * We want signal handlers to remain installed, and we want
 * signals (except SIGALRM) to restart system calls which they
 * interrupt. This is how bsd signals work, and what we'd like.
 * This function is essentially example 10.12 from Stevens'
 * _Advanced Programming in the Unix Environment_
 */
Sigfunc
install_sig_handler(int signo, Sigfunc func)
{
#ifdef HAS_SIGACTION
  struct sigaction act, oact;
  act.sa_handler = func;
  sigemptyset(&act.sa_mask);
  act.sa_flags = 0;
#ifdef SA_RESTART
  act.sa_flags |= SA_RESTART;
#endif
  if (sigaction(signo, &act, &oact) < 0)
    return SIG_ERR;
  return oact.sa_handler;
#else				/* No sigaction, drat. */
  return signal(signo, func);
#endif
}

void
reload_sig_handler(int signo __attribute__ ((__unused__)),
		   Sigfunc func __attribute__ ((__unused__)))
{
#if !(defined(HAS_SIGACTION) || defined(SIGNALS_KEPT))
  signal(signo, func);
#endif
}

void
ignore_signal(int signo)
{
#ifdef HAS_SIGACTION
  struct sigaction act;
  act.sa_handler = SIG_IGN;
  sigemptyset(&act.sa_mask);
  act.sa_flags = 0;
  sigaction(signo, &act, NULL);
#else				/* No sigaction, drat. */
  signal(signo, SIG_IGN);
#endif
}

/* These don't really work right without sigprocmask(), but we try */
void
block_a_signal(int signo)
{
#ifdef HAS_SIGPROCMASK
  sigset_t mask;
  sigemptyset(&mask);
  sigaddset(&mask, signo);
  sigprocmask(SIG_BLOCK, &mask, NULL);
#else
  if (signo > 0 && signo < NSIG)
    saved_handlers[signo] = signal(signo, SIG_IGN);
#endif
}

void
unblock_a_signal(int signo)
{
#ifdef HAS_SIGPROCMASK
  sigset_t mask;
  if (signo >= 0 && signo < NSIG) {
    sigemptyset(&mask);
    sigaddset(&mask, signo);
    sigprocmask(SIG_UNBLOCK, &mask, NULL);
  }
#else
  if (signo >= 0 && signo < NSIG)
    signal(signo, saved_handlers[signo]);
#endif
}

void
block_signals(void)
{
#ifdef HAS_SIGPROCMASK
  sigset_t mask;
  sigfillset(&mask);
  sigprocmask(SIG_BLOCK, &mask, NULL);
#else
  int i;
  for (i = 0; i < NSIG; i++)
    signal(i, SIG_IGN);
#endif
}
