
/* strutil.c */

/* String utilities */
#include "config.h"

#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <limits.h>
#include "copyrite.h"
#include "conf.h"
#include "case.h"
#include "ansi.h"
#include "pueblo.h"
#include "parse.h"
#include "externs.h"
#ifdef MEM_CHECK
#include "memcheck.h"
#endif
#include "mymalloc.h"
#include "confmagic.h"

#include "log.h"

char *next_token(char *str, char sep);
static int format_long(long val, char *buff, char **bp, int maxlen);
char *mush_strndup(const char *src, size_t len, const char *check);

/* Duplicate the first len characters of s */
char *
mush_strndup(const char *src, size_t len, const char *check)
{
  char *copy;
  size_t rlen = strlen(src);

  if (rlen < len)
    len = rlen;

  copy = mush_malloc(len + 1, check);
  if (copy) {
    memcpy(copy, src, len);
    copy[len] = '\0';
  }

  return copy;
}

char *
mush_strdup(const char *s, const char *check __attribute__ ((__unused__)))
{
  char *x;

#ifdef HAS_STRDUP
  x = strdup(s);
#ifdef MEM_CHECK
  if (x)
    add_check(check);
#endif
#else

  Size_t len = strlen(s) + 1;
  x = mush_malloc(len, check);
  if (x)
    memcpy(x, s, len);
#endif
  return x;
}

/* Return the string chopped at lim characters */
char *
chopstr(const char *str, size_t lim)
{
  static char tbuf1[BUFFER_LEN];
  if (strlen(str) <= lim)
    return (char *) str;
  strncpy(tbuf1, str, lim);
  tbuf1[lim] = '\0';
  return tbuf1;
}


#ifndef HAS_STRCASECMP
#ifndef WIN32
int
strcasecmp(const char *s1, const char *s2)
{
  while (*s1 && *s2 && DOWNCASE(*s1) == DOWNCASE(*s2))
    s1++, s2++;

  return (DOWNCASE(*s1) - DOWNCASE(*s2));
}

int
strncasecmp(const char *s1, const char *s2, Size_t n)
{
  for (; 0 < n; ++s1, ++s2, --n)
    if (DOWNCASE(*s1) != DOWNCASE(*s2))
      return DOWNCASE(*s1) - DOWNCASE(*s2);
    else if (*s1 == 0)
      return 0;
  return 0;

}
#endif				/* !WIN32 */
#endif				/* !HAS_STRCASECMP */

int
string_prefix(const char *string, const char *prefix)
{
  if (!string || !prefix)
    return 0;
  while (*string && *prefix && DOWNCASE(*string) == DOWNCASE(*prefix))
    string++, prefix++;
  return *prefix == '\0';
}

/* accepts only nonempty matches starting at the beginning of a word */
const char *
string_match(const char *src, const char *sub)
{
  if (!src || !sub)
    return 0;

  if (*sub != '\0') {
    while (*src) {
      if (string_prefix(src, sub))
	return src;
      /* else scan to beginning of next word */
      while (*src && (isalpha((unsigned char) *src)
		      || isdigit((unsigned char) *src)))
	src++;
      while (*src && !isalpha((unsigned char) *src)
	     && !isdigit((unsigned char) *src))
	src++;
    }
  }
  return 0;
}

char *
strupper(const char *s)
{
  static char buf1[BUFFER_LEN];
  char *p;

  if (!s || !*s) {
    buf1[0] = '\0';
    return buf1;
  }
  strcpy(buf1, s);
  for (p = buf1; *p; p++)
    *p = UPCASE(*p);
  return buf1;
}

char *
strlower(const char *s)
{
  static char buf1[BUFFER_LEN];
  char *p;

  if (!s || !*s) {
    buf1[0] = '\0';
    return buf1;
  }
  strcpy(buf1, s);
  for (p = buf1; *p; p++)
    *p = DOWNCASE(*p);
  return buf1;
}

char *
upcasestr(char *s)
{
  /* modifies a string in-place to be upper-case */
  char *p;
  for (p = s; p && *p; p++)
    *p = UPCASE(*p);
  return s;
}

int
safe_accent(const char *base, const char *tmplate, size_t len, char *buff,
	    char **bp)
{
  /* base and tmplate must be the same length */
  size_t n;
  unsigned char c;

  for (n = 0; n < len; n++) {
    switch (base[n]) {
    case 'A':
      switch (tmplate[n]) {
      case '`':
	c = 192;
	break;
      case '\'':
	c = 193;
	break;
      case '^':
	c = 194;
	break;
      case '~':
	c = 195;
	break;
      case ':':
	c = 196;
	break;
      case 'o':
	c = 197;
	break;
      default:
	c = 'A';
      }
      break;
    case 'a':
      switch (tmplate[n]) {
      case '`':
	c = 224;
	break;
      case '\'':
	c = 225;
	break;
      case '^':
	c = 226;
	break;
      case '~':
	c = 227;
	break;
      case ':':
	c = 228;
	break;
      case 'o':
	c = 229;
	break;
      default:
	c = 'a';
      }
      break;
    case 'C':
      if (tmplate[n] == ',')
	c = 199;
      else
	c = 'C';
      break;
    case 'c':
      if (tmplate[n] == ',')
	c = 231;
      else
	c = 'c';
      break;
    case 'E':
      switch (tmplate[n]) {
      case '`':
	c = 200;
	break;
      case '\'':
	c = 201;
	break;
      case '^':
	c = 202;
	break;
      case ':':
	c = 203;
	break;
      default:
	c = 'E';
      }
      break;
    case 'e':
      switch (tmplate[n]) {
      case '`':
	c = 232;
	break;
      case '\'':
	c = 233;
	break;
      case '^':
	c = 234;
	break;
      case ':':
	c = 235;
	break;
      default:
	c = 'e';
      }
      break;
    case 'I':
      switch (tmplate[n]) {
      case '`':
	c = 204;
	break;
      case '\'':
	c = 205;
	break;
      case '^':
	c = 206;
	break;
      case ':':
	c = 207;
	break;
      default:
	c = 'I';
      }
      break;
    case 'i':
      switch (tmplate[n]) {
      case '`':
	c = 236;
	break;
      case '\'':
	c = 237;
	break;
      case '^':
	c = 238;
	break;
      case ':':
	c = 239;
	break;
      default:
	c = 'i';
      }
      break;
    case 'N':
      if (tmplate[n] == '~')
	c = 209;
      else
	c = 'N';
      break;
    case 'n':
      if (tmplate[n] == '~')
	c = 241;
      else
	c = 'n';
      break;
    case 'O':
      switch (tmplate[n]) {
      case '`':
	c = 210;
	break;
      case '\'':
	c = 211;
	break;
      case '^':
	c = 212;
	break;
      case '~':
	c = 213;
	break;
      case ':':
	c = 214;
	break;
      default:
	c = 'O';
      }
      break;
    case 'o':
      switch (tmplate[n]) {
      case '&':
	c = 240;
	break;
      case '`':
	c = 242;
	break;
      case '\'':
	c = 243;
	break;
      case '^':
	c = 244;
	break;
      case '~':
	c = 245;
	break;
      case ':':
	c = 246;
	break;
      default:
	c = 'o';
      }
      break;
    case 'U':
      switch (tmplate[n]) {
      case '`':
	c = 217;
	break;
      case '\'':
	c = 218;
	break;
      case '^':
	c = 219;
	break;
      case ':':
	c = 220;
	break;
      default:
	c = 'U';
      }
      break;
    case 'u':
      switch (tmplate[n]) {
      case '`':
	c = 249;
	break;
      case '\'':
	c = 250;
	break;
      case '^':
	c = 251;
	break;
      case ':':
	c = 252;
	break;
      default:
	c = 'u';
      }
      break;
    case 'Y':
      if (tmplate[n] == '\'')
	c = 221;
      else
	c = 'Y';
      break;
    case 'y':
      if (tmplate[n] == '\'')
	c = 253;
      else if (tmplate[n] == ':')
	c = 255;
      else
	c = 'y';
      break;
    case '?':
      if (tmplate[n] == 'u')
	c = 191;
      else
	c = '?';
      break;
    case '!':
      if (tmplate[n] == 'u')
	c = 161;
      else
	c = '!';
      break;
    case '<':
      if (tmplate[n] == '"')
	c = 171;
      else
	c = '<';
      break;
    case '>':
      if (tmplate[n] == '"')
	c = 187;
      else
	c = '>';
      break;
    case 's':
      if (tmplate[n] == 'B')
	c = 223;
      else
	c = 's';
      break;
    case 'p':
      if (tmplate[n] == '|')
	c = 254;
      else
	c = 'p';
      break;
    case 'P':
      if (tmplate[n] == '|')
	c = 222;
      else
	c = 'P';
      break;
    case 'D':
      if (tmplate[n] == '-')
	c = 208;
      else
	c = 'D';
      break;
    default:
      c = base[n];
    }
    if (isprint(c)) {
      if (safe_chr((char) c, buff, bp))
	return 1;
    } else {
      if (safe_chr(base[n], buff, bp))
	return 1;
    }
  }
  return 0;
}


/* Macros used by the safe_XXX functions to append a
 *       string c to the end of buff, starting at *bp.
*/
#define APPEND_ARGS int len, blen, clen
#define APPEND_TO_BUF \
  /* Trivial cases */  \
  if (c[0] == '\0') \
    return 0; \
  /* The array is at least two characters long here */ \
  if (c[1] == '\0') \
    return safe_chr(c[0], buff, bp); \
  len = strlen(c); \
  blen = *bp - buff; \
  if (blen > (BUFFER_LEN - 1)) \
    return len; \
  if ((len + blen) <= (BUFFER_LEN - 1)) \
    clen = len; \
  else \
    clen = (BUFFER_LEN - 1) - blen; \
  memcpy(*bp, c, clen); \
  *bp += clen; \
  return len - clen


int
safe_format(char *buff, char **bp, const char *fmt, ...)
{
  APPEND_ARGS;
#ifdef HAS_VSNPRINTF
  char c[BUFFER_LEN];
#else
  char c[BUFFER_LEN * 3];
#endif
  va_list args;

  va_start(args, fmt);

#ifdef HAS_VSNPRINTF
  vsnprintf(c, sizeof c, fmt, args);
#else
  vsprintf(c, fmt, args);
#endif
  c[BUFFER_LEN - 1] = '\0';
  va_end(args);

  APPEND_TO_BUF;
}

int
safe_integer(int i, char *buff, char **bp)
{
  /* Adds a int to a string, being careful not to overflow buffer */
  return format_long(i, buff, bp, BUFFER_LEN);
}

int
safe_uinteger(unsigned int i, char *buff, char **bp)
{
  /* Adds an unsigned int to a string */
  return safe_str(unparse_uinteger(i), buff, bp);
}

int
safe_integer_sbuf(int i, char *buff, char **bp)
{
  /* Adds a int to a string, being careful not to overflow buffer */
  return format_long(i, buff, bp, SBUF_LEN);
}

int
safe_dbref(dbref d, char *buff, char **bp)
{
  /* Adds a dbref to a string, being careful not to overflow buffer and 
     avoiding writing partial dbrefs */
  char *saved = *bp;
  if (safe_chr('#', buff, bp)) {
    *bp = saved;
    return 1;
  }
  if (format_long(d, buff, bp, BUFFER_LEN)) {
    *bp = saved;
    return 1;
  }
  return 0;
}


int
safe_number(NVAL n, char *buff, char **bp)
{
  /* Adds a number to a string, being careful not to overflow buffer */
  const char *c;
  APPEND_ARGS;

  c = unparse_number(n);
  APPEND_TO_BUF;
}

int
safe_str(const char *c, char *buff, char **bp)
{
  /* copies a string into a buffer, making sure there's no overflow. */
  APPEND_ARGS;

  if (!c || !*c)
    return 0;

  APPEND_TO_BUF;

}

int
safe_str_space(const char *c, char *buff, char **bp)
{
  /* copies a string into a buffer, making sure there's no overflow.
   * puts it in quotes if there's space in it. */
  APPEND_ARGS;
  char *saved = *bp;

  if (!c || !*c)
    return 0;

  if (strchr(c, ' ')) {
    if (safe_chr('"', buff, bp) || safe_str(c, buff, bp) ||
	safe_chr('"', buff, bp)) {
      *bp = saved;
      return 1;
    }
    return 0;
  } else {
    APPEND_TO_BUF;
  }
}


int
safe_strl(const char *s, int len, char *buff, char **bp)
{
  int blen, clen;

  if (!s || !*s)
    return 0;
  if (len == 1)
    return safe_chr(*s, buff, bp);

  blen = *bp - buff;
  if (blen > BUFFER_LEN - 2)
    return len;
  if ((len + blen) <= BUFFER_LEN - 2)
    clen = len;
  else
    clen = BUFFER_LEN - 2 - blen;
  memcpy(*bp, s, clen);
  *bp += clen;
  return len - clen;
}

int
safe_fill(char x, size_t n, char *buff, char **bp)
{
  size_t blen;
  int ret = 0;

  if (n == 0)
    return 0;
  else if (n == 1)
    return safe_chr(x, buff, bp);

  if (n > BUFFER_LEN - 1)
    n = BUFFER_LEN - 1;

  blen = BUFFER_LEN - (*bp - buff);

  if (blen < n) {
    n = blen;
    ret = 1;
  }
  memset(*bp, x, n);
  *bp += n;

  return ret;
}

#undef APPEND_ARGS
#undef APPEND_TO_BUF

/* skip_space and seek_char are essentially right out of the 2.0 code */

char *
skip_space(const char *s)
{
  /* returns pointer to the next non-space char in s, or NULL if s == NULL
   * or *s == NULL or s has only spaces.
   */

  char *c = (char *) s;
  while (c && *c && isspace((unsigned char) *c))
    c++;
  return c;
}

char *
seek_char(const char *s, char c)
{
  /* similar to strchr(). returns a pointer to the next char in s which
   * matches c, or a pointer to the terminating null at the end of s.
   */

  char *p = (char *) s;
  while (p && *p && (*p != c))
    p++;
  return p;
}

int
u_strlen(const unsigned char *s)
{
  return strlen((const char *) s);
}

unsigned char *
u_strcpy(unsigned char *target, const unsigned char *source)
{
  return (unsigned char *) strcpy((char *) target, (const char *) source);
}

char *
replace_string(const char *old, const char *newbit, const char *string)
{
  char *result, *r;
  Size_t len, newlen;

  r = result = mush_malloc(BUFFER_LEN, "replace_string.buff");
  if (!result)
    panic(T("Couldn't allocate memory in replace_string!"));

  len = strlen(old);
  newlen = strlen(newbit);

  while (*string) {
    char *s = strstr(string, old);
    if (s) {			/* Match found! */
      safe_strl(string, s - string, result, &r);
      safe_strl(newbit, newlen, result, &r);
      string = s + len;
    } else {
      safe_str(string, result, &r);
      break;
    }
  }
  *r = '\0';
  return result;
}

const char *standard_tokens[2] = { "##", "#@" };

/* Replace two tokens in a string at once. All-around better than calling
 * replace_string() twice
 */
char *
replace_string2(const char *old[2], const char *newbits[2], const char *string)
{
  char *result, *rp;
  char firsts[3] = { '\0', '\0', '\0' };
  Size_t oldlens[2], newlens[2];

  if (!string)
    return NULL;

  rp = result = mush_malloc(BUFFER_LEN, "replace_string.buff");
  if (!result)
    panic(T("Couldn't allocate memory in replace_string2!"));

  firsts[0] = old[0][0];
  firsts[1] = old[1][0];

  oldlens[0] = strlen(old[0]);
  oldlens[1] = strlen(old[1]);
  newlens[0] = strlen(newbits[0]);
  newlens[1] = strlen(newbits[1]);

  while (*string) {
    Size_t skip = strcspn(string, firsts);
    if (skip) {
      safe_strl(string, skip, result, &rp);
      string += skip;
    }
    if (strncmp(string, old[0], oldlens[0]) == 0) {	/* Copy the first */
      safe_strl(newbits[0], newlens[0], result, &rp);
      string += oldlens[0];
    } else if (strncmp(string, old[1], oldlens[1]) == 0) {	/* The second */
      safe_strl(newbits[1], newlens[1], result, &rp);
      string += oldlens[1];
    } else {
      safe_chr(*string, result, &rp);
      string++;
    }
  }

  *rp = '\0';
  return result;

}

char *
trim_space_sep(char *str, char sep)
{
  /* Trim leading and trailing spaces if we've got a space separator. */

  char *p;

  if (sep != ' ')
    return str;
  /* Skip leading spaces */
  str += strspn(str, " ");
  for (p = str; *p; p++) ;
  /* And trailing */
  for (p--; (*p == ' ') && (p > str); p--) ;
  p++;
  *p = '\0';
  return str;
}

char *
next_token(char *str, char sep)
{
  /* move pointer to start of the next token */

  while (*str && (*str != sep))
    str++;
  if (!*str)
    return NULL;
  str++;
  if (sep == ' ') {
    while (*str == sep)
      str++;
  }
  return str;
}

char *
split_token(char **sp, char sep)
{
  /* Get next token from string as a null-terminated string, depending
   * on the separator character. This destructively modifies the string.
   * Code from 2.0.
   */

  char *str, *save;

  save = str = *sp;
  if (!str) {
    *sp = NULL;
    return NULL;
  }
  while (*str && (*str != sep))
    str++;
  if (*str) {
    *str++ = '\0';
    if (sep == ' ') {
      while (*str == sep)
	str++;
    }
  } else {
    str = NULL;
  }
  *sp = str;
  return save;
}

int
do_wordcount(char *str, char sep)
{
  /* count the number of words in a string */
  int n;

  if (!*str)
    return 0;
  for (n = 0; str; str = next_token(str, sep), n++) ;
  return n;
}

/* Pretty much straight from Tiny 2.2, we see if str and target
 * match in the first min chars
 */
int
minmatch(const char *str, const char *target, int min)
{
  while (*str && *target && (DOWNCASE(*str) == DOWNCASE(*target))) {
    str++;
    target++;
    min--;
  }
  if (*str)
    return 0;
  if (!*target)
    return 1;
  return ((min <= 0) ? 1 : 0);
}

/* Strlen that ignores ansi and HTML sequences */

int
ansi_strlen(const char *p)
{
  int i = 0;

  if (!p)
    return 0;

  if (!ANSI_JUSTIFY)
    return strlen(p);

  while (*p) {
    if (*p == ESC_CHAR) {
      while ((*p) && (*p != 'm'))
	p++;
    } else if (*p == TAG_START) {
      while ((*p) && (*p != TAG_END))
	p++;
    } else {
      i++;
    }
    p++;
  }
  return i;
}

/* Returns true length of string up to numchars visible characters. 
 */
int
ansi_strnlen(const char *p, size_t numchars)
{
  int i = 0;

  if (!p)
    return 0;
  while (*p && numchars > 0) {
    if (*p == ESC_CHAR) {
      while ((*p) && (*p != 'm')) {
	p++;
	i++;
      }
    } else if (*p == TAG_START) {
      while ((*p) && (*p != TAG_END)) {
	p++;
	i++;
      }
    } else
      numchars--;
    i++;
    p++;
  }
  return i;
}

/* Given a string, a word, and a separator, remove first occurence
 * of the word from the string. Destructive.
 */
char *
remove_word(char *list, char *word, char sep)
{
  char *sp;
  char *bp;
  static char buff[BUFFER_LEN];

  bp = buff;
  sp = split_token(&list, sep);
  if (!strcmp(sp, word)) {
    sp = split_token(&list, sep);
    safe_str(sp, buff, &bp);
  } else {
    safe_str(sp, buff, &bp);
    while (list && strcmp(sp = split_token(&list, sep), word)) {
      safe_chr(sep, buff, &bp);
      safe_str(sp, buff, &bp);
    }
  }
  while (list) {
    sp = split_token(&list, sep);
    safe_chr(sep, buff, &bp);
    safe_str(sp, buff, &bp);
  }
  *bp = '\0';
  return buff;
}

char *
next_in_list(const char **head)
{
  int paren = 0;
  static char buf[BUFFER_LEN];
  char *p = buf;

  while (**head == ' ')
    (*head)++;

  if (**head == '"') {
    (*head)++;
    paren = 1;
  }

  /* Copy it char by char until you hit a " or, if not in a
   * paren, a space
   */
  while (**head && (paren || (**head != ' ')) && (**head != '"')) {
    safe_chr(**head, buf, &p);
    (*head)++;
  }

  if (paren && **head)
    (*head)++;

  safe_chr('\0', buf, &p);
  return buf;

}

/* Strip all ansi and html markup from a string */
char *
remove_markup(const char *orig, size_t * s_len)
{
  static char buff[BUFFER_LEN];
  char *bp = buff;
  const char *q;
  size_t len = 0;

  if (!orig)
    return NULL;

  for (q = orig; *q;) {
    switch (*q) {
    case ESC_CHAR:
      /* Skip over ansi */
      while (*q && *q++ != 'm') ;
      break;
    case TAG_START:
      /* Skip over HTML */
      while (*q && *q++ != TAG_END) ;
      break;
    default:
      safe_chr(*q++, buff, &bp);
      len++;
    }
  }
  *bp = '\0';
  if (s_len)
    *s_len = len + 1;
  return buff;
}


/* Append an int to the string. Returns a true value on failure.
 * This will someday take extra arguments for use with our version 
 * of snprintf. Please try not to use it.
 */
/* maxlen = total length of string.
   buf[maxlen - 1] = place where \0 will go.
   buf[maxlen - 2] = last visible character.
*/
static int
format_long(long val, char *buff, char **bp, int maxlen)
{
  char stack[128];		/* Even a negative 64 bit number will only be 21
				   digits or so max. This should be plenty of
				   buffer room. */
  char *current;
  int size = 0, neg = 0;
  ldiv_t r;

  /* Sanity checks */
  if (!bp || !buff || !*bp)
    return 1;
  if (*bp - buff >= maxlen - 1)
    return 1;

  if (val < 0) {
    neg = 1;
    val = -val;
    if (val < 0) {
      /* -LONG_MIN == LONG_MIN on 2's complement systems. Take the
         easy way out since this value is rarely encountered. */
      sprintf(stack, "%ld", val);
      return safe_str(stack, buff, bp);
    }

  }

  current = stack + sizeof(stack);

  /* Take the rightmost digit, and push it onto the stack, then
   * integer divide by 10 to get to the next digit. */
  r.quot = val;
  do {
    /* ldiv(x, y) does x/y and x%y at the same time (both of
     * which we need).
     */
    r = ldiv(r.quot, 10);
    *(--current) = (unsigned char) r.rem + '0';
  } while (r.quot);

  /* Add the negative sign if needed. */
  if (neg)
    *(--current) = '-';

  /* The above puts the number on the stack.  Now we need to put
   * it in the buffer.  If there's enough room, use Duff's Device
   * for speed, otherwise do it one char at a time.
   */

  size = stack + sizeof(stack) - current;

  /* if (size < (int) ((buff + maxlen - 1) - *bp)) { */
  if (((int) (*bp - buff)) + size < maxlen - 2) {
    switch (size % 8) {
    case 0:
      while (current < stack + sizeof(stack)) {
	*((*bp)++) = *(current++);
    case 7:
	*((*bp)++) = *(current++);
    case 6:
	*((*bp)++) = *(current++);
    case 5:
	*((*bp)++) = *(current++);
    case 4:
	*((*bp)++) = *(current++);
    case 3:
	*((*bp)++) = *(current++);
    case 2:
	*((*bp)++) = *(current++);
    case 1:
	*((*bp)++) = *(current++);
      }
    }
  } else {
    while (current < stack + sizeof(stack)) {
      if (*bp - buff >= maxlen - 1) {
	return 1;
      }
      *((*bp)++) = *(current++);
    }
  }

  return 0;
}

#if defined(HAS_STRXFRM) && !defined(WIN32)
int
strncoll(const char *s1, const char *s2, size_t t)
{
  char *d1, *d2, *ns1, *ns2;
  int result;
  size_t s1_len, s2_len;

  ns1 = mush_malloc(t + 1, "string");
  ns2 = mush_malloc(t + 1, "string");
  memcpy(ns1, s1, t);
  ns1[t] = '\0';
  memcpy(ns2, s2, t);
  ns2[t] = '\0';
  s1_len = strxfrm(NULL, ns1, 0) + 1;
  s2_len = strxfrm(NULL, ns2, 0) + 1;

  d1 = mush_malloc(s1_len + 1, "string");
  d2 = mush_malloc(s2_len + 1, "string");
  (void) strxfrm(d1, ns1, s1_len);
  (void) strxfrm(d2, ns2, s2_len);
  result = strcmp(d1, d2);
  mush_free(d1, "string");
  mush_free(d2, "string");
  return result;
}

int
strcasecoll(const char *s1, const char *s2)
{
  char *d1, *d2;
  int result;
  size_t s1_len, s2_len;

  s1_len = strxfrm(NULL, s1, 0) + 1;
  s2_len = strxfrm(NULL, s2, 0) + 1;

  d1 = mush_malloc(s1_len, "string");
  d2 = mush_malloc(s2_len, "string");
  (void) strxfrm(d1, strupper(s1), s1_len);
  (void) strxfrm(d2, strupper(s2), s2_len);
  result = strcmp(d1, d2);
  mush_free(d1, "string");
  mush_free(d2, "string");
  return result;
}

int
strncasecoll(const char *s1, const char *s2, size_t t)
{
  char *d1, *d2, *ns1, *ns2;
  int result;
  size_t s1_len, s2_len;

  ns1 = mush_malloc(t + 1, "string");
  ns2 = mush_malloc(t + 1, "string");
  memcpy(ns1, s1, t);
  ns1[t] = '\0';
  memcpy(ns2, s2, t);
  ns2[t] = '\0';
  s1_len = strxfrm(NULL, ns1, 0) + 1;
  s2_len = strxfrm(NULL, ns2, 0) + 1;

  d1 = mush_malloc(s1_len, "string");
  d2 = mush_malloc(s2_len, "string");
  (void) strxfrm(d1, strupper(ns1), s1_len);
  (void) strxfrm(d2, strupper(ns2), s2_len);
  result = strcmp(d1, d2);
  mush_free(d1, "string");
  mush_free(d2, "string");
  return result;
}
#endif				/* HAS_STRXFRM && !WIN32 */

char *
skip_leading_ansi(const char *p)
{
  if (!p)
    return NULL;
  while (*p == ESC_CHAR || *p == TAG_START) {
    if (*p == ESC_CHAR) {
      while (*p && *p != 'm')
	p++;
    } else {			/* TAG_START */
      while (*p && *p != TAG_END)
	p++;
    }
    if (*p)
      p++;
  }
  return (char *) p;

}

ansi_string *
parse_ansi_string(const char *src)
{
  ansi_string *data;
  char *y, *current = NULL;
  Size_t p = 0;

  if (!src)
    return NULL;

  data = mush_malloc(sizeof *data, "ansi_string");
  if (!data)
    return NULL;

  data->len = ansi_strlen(src);

  while (*src) {
    y = skip_leading_ansi(src);
    if (y != src) {
      if (current)
	mush_free(current, "markup_codes");
      current = mush_strndup(src, y - src, "markup_codes");
      src = y;
    }
    if (current)
      data->codes[p] = mush_strdup(current, "markup_codes");
    else
      data->codes[p] = NULL;
    data->text[p] = *src;
    if (*src)
      src++;
    p++;
  }
  data->text[p] = '\0';

  while (p <= data->len) {
    data->codes[p] = NULL;
    p++;
  }

  if (current)
    mush_free(current, "markup_codes");

  return data;
}


void
populate_codes(ansi_string * as)
{
  size_t p;
  char *current = NULL;

  if (!as)
    return;

  for (p = 0; p < as->len; p++)
    if (as->codes[p]) {
      if (current)
	mush_free(current, "markup_codes");
      current = mush_strdup(as->codes[p], "markup_codes");
    } else {
      if (!current)
	current = mush_strdup(ANSI_NORMAL, "markup_codes");
      as->codes[p] = mush_strdup(current, "markup_codes");
    }
  if (current)
    mush_free(current, "markup_codes");
}

void
depopulate_codes(ansi_string * as)
{
  size_t p, m;
  int normal = 1;

  if (!as)
    return;

  for (p = 0; p <= as->len; p++) {
    if (as->codes[p]) {
      if (normal) {
	if (strcmp(as->codes[p], ANSI_NORMAL) == 0) {
	  mush_free(as->codes[p], "markup_codes");
	  as->codes[p] = NULL;
	  continue;
	} else {
	  normal = 0;
	}
      }

      m = p;
      for (p++; p < as->len; p++) {
	if (as->codes[p] && strcmp(as->codes[p], as->codes[m]) == 0) {
	  mush_free(as->codes[p], "markup_codes");
	  as->codes[p] = NULL;
	} else {
	  p--;
	  break;
	}
      }
    }
  }
}

static int is_ansi_code(const char *s);
static int is_start_html_code(const char *s) __attribute__ ((__unused__));
static int is_end_html_code(const char *s);
#define is_end_ansi_code(s) (!strcmp((s),ANSI_NORMAL))


static int
is_ansi_code(const char *s)
{
  return s && *s == ESC_CHAR;
}

static int
is_start_html_code(const char *s)
{
  return s && *s == TAG_START && *(s + 1) != '/';
}

static int
is_end_html_code(const char *s)
{
  return s && *s == TAG_START && *(s + 1) == '/';
}

void
free_ansi_string(ansi_string * as)
{
  int p;

  if (!as)
    return;
  for (p = as->len; p >= 0; p--) {
    if (as->codes[p])
      mush_free(as->codes[p], "markup_codes");
  }
  mush_free(as, "ansi_string");
}

int
safe_ansi_string(ansi_string * as, size_t start, size_t len, char *buff,
		 char **bp)
{
  int p, q;
  int in_ansi = 0;
  int in_html = 0;

  if (!as)
    return 1;

  depopulate_codes(as);

  if (start > as->len || len == 0 || as->len == 0)
    return safe_str("", buff, bp);

  /* Find the starting codes by working our way backward until we
   * reach some opening codes, and then working our way back from there
   * until we hit a non-opening code or non-code 
   */
  p = start;
  while ((p >= 0) && (as->codes[p] == NULL))
    p--;
  /* p is now either <0 or pointing to a code */
  if ((p >= 0) && !is_end_html_code(as->codes[p]) &&
      !is_end_ansi_code(as->codes[p])) {
    /* p is now pointing to a starting code */
    q = p;
    while ((q >= 0) && as->codes[q] && !is_end_html_code(as->codes[q]) &&
	   !is_end_ansi_code(as->codes[q])) {
      if (is_ansi_code(as->codes[q]))
	in_ansi = 1;
      else if (is_start_html_code(as->codes[q]))
	in_html++;
      q--;
    }
    p = q + 1;
    /* p is now pointing to the first starting code, and we know if we're
     * in ansi, html, or both. We also know how many html tags have been
     * opened.
     */
  }

  /* Copy the text. The right thing to do now would be to have a stack
   * of open html tags and clear in_html once all of the tags have
   * been closed. We don't quite do that, alas.
   */
  for (p = (int) start; p < (int) (start + len) && p < (int) as->len; p++) {
    if (as->codes[p]) {
      if (safe_str(as->codes[p], buff, bp))
	return 1;
      if (is_end_ansi_code(as->codes[p]))
	in_ansi = 0;
      else if (is_ansi_code(as->codes[p]))
	in_ansi = 1;
      if (is_end_html_code(as->codes[p]))
	in_html--;
      else if (is_start_html_code(as->codes[p]))
	in_html++;
    }
    if (safe_chr(as->text[p], buff, bp))
      return 1;
  }

  /* Output (only) closing codes if needed. */
  while (p <= (int) as->len) {
    if (!in_ansi && !in_html)
      break;
    if (as->codes[p]) {
      if (is_end_ansi_code(as->codes[p])) {
	in_ansi = 0;
	if (safe_str(as->codes[p], buff, bp))
	  return 1;
      } else if (is_end_html_code(as->codes[p])) {
	in_html--;
	if (safe_str(as->codes[p], buff, bp))
	  return 1;
      }
    }
    p++;
  }
  if (in_ansi)
    safe_str(ANSI_NORMAL, buff, bp);
  return 0;
}

/* Given the current item number in a list, whether it's the last item
 * in the list, the list's output separator, a conjunction,
 * and a punctuation mark to use between items, store the appropriate
 * inter-item stuff into the given buffer safely.
 */
void
safe_itemizer(int cur_num, int done, const char *delim, const char *conjoin,
	      const char *space, char *buff, char **bp)
{
  /* We don't do anything if it's the first one */
  if (cur_num == 1)
    return;
  /* Are we done? */
  if (done) {
    /* if so, we need a [<delim>]<space><conj> */
    if (cur_num >= 3)
      safe_str(delim, buff, bp);
    safe_str(space, buff, bp);
    safe_str(conjoin, buff, bp);
  } else {
    /* if not, we need just <delim> */
    safe_str(delim, buff, bp);
  }
  /* And then we need another <space> */
  safe_str(space, buff, bp);

}
