/**
 * \file funlist.c
 *
 * \brief List-handling functions for mushcode.
 *
 *
 */

#include "copyrite.h"

#define _GNU_SOURCE
#include <string.h>
#include <ctype.h>
#include "ansi.h"
#include "attrib.h"
#include "case.h"
#include "command.h"
#include "conf.h"
#include "dbdefs.h"
#include "externs.h"
#include "flags.h"
#include "function.h"
#include "lock.h"
#include "match.h"
#include "memcheck.h"
#include "mushdb.h"
#include "mymalloc.h"
#include "mypcre.h"
#include "notify.h"
#include "parse.h"
#include "sort.h"
#include "strutil.h"

enum itemfun_op { IF_DELETE, IF_REPLACE, IF_INSERT };
static void freearr_member(char *p);
extern const unsigned char *tables;
static int find_list_position(char *numstr, int total, bool insert);

/** Convert list to array.
 * Chops up a list of words into an array of words. The list is
 * destructively modified. The array returned consists of
 * mush_strdup'd strings.
 * \param r pointer to array to store words.
 * \param max maximum number of words to split out.
 * \param list list of words as a string.
 * \param sep separator character between list items.
 * \param nullok are null elements allowed? if not, they'll be removed
 * \return number of words split out.
 */
int
list2arr_ansi(char *r[], int max, char *list, char sep, int nullok)
{
  char *p, *lp;
  int i;
  ansi_string *as;
  char *aptr;

  /* Since ansi_string is ridiculously slow, we only use it if the string
   * actually has markup. Unfortunately, freearr(), which is called only for
   * list2arr_ansi()'d stuff, requires we malloc each item. Sigh. */
  if (!has_markup(list)) {
    int ret = list2arr(r, max, list, sep, nullok);
    for (i = 0; i < ret; i++) {
      /* This is lame, but fortunately, assignment happens after we call
       * mush_strdup. A-hehehehe. */
      r[i] = mush_strdup(r[i], "list2arr_item");
    }
    return ret;
  }

  as = parse_ansi_string(list);
  aptr = as->text;

  aptr = trim_space_sep(aptr, sep);

  lp = list;
  do {
    p = split_token(&aptr, sep);
  } while (!nullok && p && !*p);
  for (i = 0; p && (i < max); i++) {
    lp = list;
    safe_ansi_string(as, p - (as->text), strlen(p), list, &lp);
    *lp = '\0';
    r[i] = mush_strdup(list, "list2arr_item");
    do {
      p = split_token(&aptr, sep);
    } while (!nullok && p && !*p);
  }
  free_ansi_string(as);
  return i;
}

/** Convert list to array.
 * Chops up a list of words into an array of words. The list is
 * destructively modified.
 * \param r pointer to array to store words.
 * \param max maximum number of words to split out.
 * \param list list of words as a string.
 * \param sep separator character between list items.
 * \param nullok are null elements allowed? if not, they'll be removed
 * \return number of words split out.
 */
int
list2arr(char *r[], int max, char *list, char sep, int nullok)
{
  char *p;
  int i;
  char *aptr;
  size_t len;

  /* Quick-casing an empty list. */
  if (!*list) {
    return 0;
  }

  aptr = remove_markup(list, &len);

  memcpy(list, aptr, len);

  aptr = trim_space_sep(list, sep);

  do {
    p = split_token(&aptr, sep);
  } while (!nullok && p && !*p);
  for (i = 0; p && (i < max); i++) {
    r[i] = p;
    do {
      p = split_token(&aptr, sep);
    } while (!nullok && p && !*p);
  }
  return i;
}

/** Convert array to list.
 * Takes an array of words and concatenates them into a string,
 * using our safe string functions.
 * \param r pointer to array of words.
 * \param max maximum number of words to concatenate.
 * \param list string to fill with word list.
 * \param lp pointer into end of list.
 * \param sep string to use as separator between words.
 */
void
arr2list(char *r[], int max, char *list, char **lp, const char *sep)
{
  int i = 0, j = 0, seplen = 0;

  if (!max)
    return;

  if (sep && *sep)
    seplen = strlen(sep);

  for (i = 0; i < max; i++) {
    if (r[i]) {
      if (j) {
        safe_strl(sep, seplen, list, lp);
      }
      safe_str(r[i], list, lp);
      j++;
    }
  }
  **lp = '\0';
}

static void
freearr_member(char *p)
{
  mush_free(p, "list2arr_item");
}

/** Free an array generated by list2arr_ansi().
 * Takes an array of words and frees it.
 * \param r pointer to array of words.
 * \param size The number of items in the list.
 */
void
freearr(char *r[], int size)
{
  int i;
  for (i = 0; i < size; i++) {
    if (r[i])
      freearr_member(r[i]);
  }
}

/* For a list with X items, return the appropriate index for the Yth element.
 * Y may be negative, in which case we count from the end of the list.
 * When inserting, we add 1 to the result for negative indicies.
 * Lists are 1-indexed, not 0-indexed.
 * last+1th is totally a valid way of putting it. Shut up, it's late.
 * \param numstr the index to find, as a string
 * \param total the total number of elements in the list
 * \param insert Are we inserting into the list at this position?
 */
static int
find_list_position(char *numstr, int total, bool insert)
{
  int i;
  bool negative = false;

  if (!is_integer(numstr))
    return -1;

  i = parse_integer(numstr);

  if (i < 0) {
    i = (total + 1) - (i * -1);
    negative = true;
  }

  if (i < 1 || i > total) {
    if (total == 0 && insert && ((i == 0 && negative) || i == 1)) {
      /* Special case: inserting into an empty list
       * will create a one-element list */
      return 1;
    } else {
      return -1;
    }
  } else if (insert && negative)
    return i + 1;
  else
    return i;
}

/* ARGSUSED */
FUNCTION(fun_munge)
{
  /* This is a function which takes three arguments. The first is
   * an obj-attr pair referencing a u-function to be called. The
   * other two arguments are lists. The first list is passed to the
   * u-function.  The second list is then rearranged to match the
   * order of the first list as returned from the u-function.
   * This rearranged list is returned by MUNGE.
   * A fourth argument (separator) is optional.
   */
  char list1[BUFFER_LEN], *lp, rlist[BUFFER_LEN];
  char **ptrs1, **ptrs2, **results;
  char **ptrs3;
  int i, j, nptrs1, nptrs2, nresults;
  ufun_attrib ufun;
  char sep, isep[2] = {'\0', '\0'}, *osep, osepd[2] = {'\0', '\0'};
  int first;
  PE_REGS *pe_regs;

  if (!delim_check(buff, bp, nargs, args, 4, &sep))
    return;

  isep[0] = sep;
  if (nargs == 5)
    osep = args[4];
  else {
    osepd[0] = sep;
    osep = osepd;
  }

  /* find our object and attribute */
  if (!fetch_ufun_attrib(args[0], executor, &ufun, UFUN_DEFAULT))
    return;

  /* Copy the first list, since we need to pass it to two destructive
   * routines.
   */

  strcpy(list1, remove_markup(args[1], NULL));

  /* Break up the two lists into their respective elements. */

  ptrs1 = mush_calloc(MAX_SORTSIZE, sizeof(char *), "ptrarray");
  ptrs2 = mush_calloc(MAX_SORTSIZE, sizeof(char *), "ptrarray");

  /* ptrs3 is destructively modified, but it's a copy of ptrs2, so we
   * make it a straight copy of ptrs2 and freearr() on ptrs2. */
  ptrs3 = mush_calloc(MAX_SORTSIZE, sizeof(char *), "ptrarray");

  if (!ptrs1 || !ptrs2)
    mush_panic("Unable to allocate memory in fun_munge");
  nptrs1 = list2arr_ansi(ptrs1, MAX_SORTSIZE, list1, sep, 1);
  nptrs2 = list2arr_ansi(ptrs2, MAX_SORTSIZE, args[2], sep, 1);
  memcpy(ptrs3, ptrs2, MAX_SORTSIZE * sizeof(char *));

  if (nptrs1 != nptrs2) {
    safe_str(T("#-1 LISTS MUST BE OF EQUAL SIZE"), buff, bp);
    freearr(ptrs1, nptrs1);
    freearr(ptrs2, nptrs2);
    mush_free(ptrs1, "ptrarray");
    mush_free(ptrs2, "ptrarray");
    mush_free(ptrs3, "ptrarray");
    return;
  }

  /* Call the user function */
  lp = args[1];
  pe_regs = pe_regs_create(PE_REGS_ARG, "fun_munge");
  pe_regs_setenv_nocopy(pe_regs, 0, lp);
  pe_regs_setenv_nocopy(pe_regs, 1, isep);
  call_ufun(&ufun, rlist, executor, enactor, pe_info, pe_regs);
  pe_regs_free(pe_regs);
  strcpy(rlist, remove_markup(rlist, NULL));

  /* Now that we have our result, put it back into array form. Search
   * through list1 until we find the element position, then copy the
   * corresponding element from list2.  Mark used elements with
   * NULL to handle duplicates
   */
  results = mush_calloc(MAX_SORTSIZE, sizeof(char *), "ptrarray");
  if (!results)
    mush_panic("Unable to allocate memory in fun_munge");
  nresults = list2arr_ansi(results, MAX_SORTSIZE, rlist, sep, 1);

  first = 1;
  for (i = 0; i < nresults; i++) {
    for (j = 0; j < nptrs1; j++) {
      if (ptrs3[j] && !strcmp(results[i], ptrs1[j])) {
        if (first)
          first = 0;
        else
          safe_str(osep, buff, bp);
        safe_str(ptrs3[j], buff, bp);
        ptrs3[j] = NULL;
        break;
      }
    }
  }
  freearr(ptrs1, nptrs1);
  freearr(ptrs2, nptrs2);
  freearr(results, nresults);
  mush_free(ptrs1, "ptrarray");
  mush_free(ptrs2, "ptrarray");
  mush_free(ptrs3, "ptrarray");
  mush_free(results, "ptrarray");
}

/* ARGSUSED */
FUNCTION(fun_elements)
{
  /* Given a list and a list of numbers, return the corresponding
   * elements of the list. elements(ack bar eep foof yay,2 4) = bar foof
   * A separator for the first list is allowed.
   */
  int nwords, cur;
  int count = 0;
  char **ptrs;
  char *wordlist;
  char *s, *r, sep;
  char *osep, osepd[2] = {'\0', '\0'};

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;

  if (nargs == 4)
    osep = args[3];
  else {
    osepd[0] = sep;
    osep = osepd;
  }

  ptrs = mush_calloc(MAX_SORTSIZE, sizeof(char *), "ptrarray");
  wordlist = mush_malloc(BUFFER_LEN, "string");
  if (!ptrs || !wordlist)
    mush_panic("Unable to allocate memory in fun_elements");

  /* Turn the first list into an array. */
  strcpy(wordlist, args[0]);
  nwords = list2arr_ansi(ptrs, MAX_SORTSIZE, wordlist, sep, 1);

  s = trim_space_sep(args[1], ' ');

  /* Go through the second list, grabbing the numbers and finding the
   * corresponding elements.
   */

  while (s) {
    r = split_token(&s, ' ');
    cur = find_list_position(r, nwords, 0) - 1;
    if ((cur >= 0) && (cur < nwords) && ptrs[cur]) {
      if (count)
        safe_str(osep, buff, bp);
      count++;
      safe_str(ptrs[cur], buff, bp);
    }
  }
  freearr(ptrs, nwords);
  mush_free(ptrs, "ptrarray");
  mush_free(wordlist, "string");
}

/* ARGSUSED */
FUNCTION(fun_matchall)
{
  /* Check each word individually, returning the word number of all
   * that match. If none match, return an empty string.
   */

  int wcount;
  char *r, *s, *b, sep;
  char *osep, osepd[2] = {'\0', '\0'};

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;

  if (nargs == 4)
    osep = args[3];
  else {
    osepd[0] = sep;
    osep = osepd;
  }

  wcount = 1;
  s = trim_space_sep(args[0], sep);
  b = *bp;
  do {
    r = split_token(&s, sep);
    if (quick_wild(args[1], r)) {
      if (*bp != b)
        safe_str(osep, buff, bp);
      safe_integer(wcount, buff, bp);
    }
    wcount++;
  } while (s);
}

/* ARGSUSED */
FUNCTION(fun_graball)
{
  /* Check each word individually, returning all that match.
   * If none match, return an empty string.  This is to grab()
   * what matchall() is to match().
   */

  char *r, *s, *b, sep;
  char *osep, osepd[2] = {'\0', '\0'};
  ansi_string *as = NULL;

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;

  if (nargs == 4)
    osep = args[3];
  else {
    osepd[0] = sep;
    osep = osepd;
  }

  s = trim_space_sep(args[0], sep);
  if (has_markup(s)) {
    as = parse_ansi_string(s);
    s = as->text;
  }
  b = *bp;
  do {
    r = split_token(&s, sep);
    if (quick_wild(args[1], r)) {
      if (*bp != b)
        safe_str(osep, buff, bp);
      if (as) {
        safe_ansi_string(as, r - as->text, strlen(r), buff, bp);
      } else {
        safe_str(r, buff, bp);
      }
    }
  } while (s);
  if (as)
    free_ansi_string(as);
}

/* ARGSUSED */
FUNCTION(fun_fold)
{
  /* iteratively evaluates an attribute with a list of arguments and
   * optional base case. With no base case, the first list element is
   * passed as %0, and the second as %1. The attribute is then evaluated
   * with these args. The result is then used as %0, and the next arg as
   * %1. Repeat until no elements are left in the list. The base case
   * can provide a starting point.
   */

  ufun_attrib ufun;
  PE_REGS *pe_regs;
  char sep;
  int funccount, per;
  char base[BUFFER_LEN];
  char result[BUFFER_LEN];
  char *list[MAX_SORTSIZE] = {NULL};
  int n, j, i = 0;
  const char *strtwo = pe_regs_intname(2);

  if (!delim_check(buff, bp, nargs, args, 4, &sep))
    return;

  if (!fetch_ufun_attrib(args[0], executor, &ufun, UFUN_DEFAULT))
    return;

  n = list2arr_ansi(list, MAX_SORTSIZE, args[1], sep, 1);

  /* If we have three or more arguments, the third one is the base case */
  if (nargs >= 3) {
    mush_strncpy(base, args[2], sizeof base);
  } else {
    if (!n) {
      base[0] = '\0';
      i++;
    } else {
      mush_strncpy(base, list[i++], sizeof base);
    }
  }
  pe_regs = pe_regs_create(PE_REGS_ARG, "fun_fold");
  pe_regs_setenv_nocopy(pe_regs, 0, base);
  pe_regs_setenv_nocopy(pe_regs, 1, list[i++]);
  pe_regs_set_int(pe_regs, PE_REGS_ARG, strtwo, 0);
  call_ufun(&ufun, result, executor, enactor, pe_info, pe_regs);

  mush_strncpy(base, result, sizeof base);

  funccount = pe_info->fun_invocations;

  /* handle the rest of the cases */
  for (j = 1; i < n; i++, j++) {
    pe_regs_setenv_nocopy(pe_regs, 1, list[i]);
    pe_regs_set_int(pe_regs, PE_REGS_ARG, strtwo, j);
    per = call_ufun(&ufun, result, executor, enactor, pe_info, pe_regs);
    if (per || (pe_info->fun_invocations >= FUNCTION_LIMIT &&
                pe_info->fun_invocations == funccount && !strcmp(base, result)))
      break;
    funccount = pe_info->fun_invocations;
    mush_strncpy(base, result, sizeof base);
  }
  pe_regs_free(pe_regs);
  freearr(list, n);
  safe_str(base, buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_itemize)
{
  /* Called in one of two ways:
   * itemize(<list>[,<delim>[,<conjunction>[,<punctuation>]]])
   * elist(<list>[,<conjunction> [,<delim> [,<output delim> [,<punctuation>]]]])
   * Either way, it takes the elements of list and:
   *  If there's just one, returns it.
   *  If there's two, returns <e1> <conjunction> <e2>
   *  If there's >2, returns <e1><punc> <e2><punc> ... <conjunction> <en>
   * Default <conjunction> is "and", default punctuation is ","
   */
  const char *outsep = " ";
  char sep = ' ';
  const char *lconj = "and";
  const char *punc = ",";
  int pos, n;
  char *words[MAX_SORTSIZE];

  if (strcmp(called_as, "ELIST") == 0) {
    /* elist ordering */
    if (!delim_check(buff, bp, nargs, args, 3, &sep))
      return;
    if (nargs > 1)
      lconj = args[1];
    if (nargs > 3)
      outsep = args[3];
    if (nargs > 4)
      punc = args[4];
  } else {
    /* itemize ordering */
    if (!delim_check(buff, bp, nargs, args, 2, &sep))
      return;
    if (nargs > 2)
      lconj = args[2];
    if (nargs > 3)
      punc = args[3];
  }
  n = list2arr_ansi(words, MAX_SORTSIZE, args[0], sep, 1);
  for (pos = 0; pos < n; pos++) {
    safe_itemizer(pos + 1, pos == (n - 1), punc, lconj, outsep, buff, bp);
    safe_str(words[pos], buff, bp);
  }
  freearr(words, n);
}

/* ARGSUSED */
FUNCTION(fun_filter)
{
  /* take a user-def function and a list, and return only those elements
   * of the list for which the function evaluates to 1.
   */

  ufun_attrib ufun;
  char *list[MAX_SORTSIZE];
  int n;
  char result[BUFFER_LEN];
  PE_REGS *pe_regs;
  char sep;
  int first;
  int check_bool = 0;
  int funccount;
  int i;
  char *osep, osepd[2] = {'\0', '\0'};

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;

  osepd[0] = sep;
  osep = (nargs >= 4) ? args[3] : osepd;

  if (strcmp(called_as, "FILTERBOOL") == 0)
    check_bool = 1;

  /* find our object and attribute */
  if (!fetch_ufun_attrib(args[0], executor, &ufun, UFUN_DEFAULT))
    return;

  /* Go through each argument */
  n = list2arr_ansi(list, MAX_SORTSIZE, args[1], sep, 1);
  first = 1;
  funccount = pe_info->fun_invocations;
  pe_regs = pe_regs_create(PE_REGS_ARG, "fun_filter");
  for (i = 4; i < nargs; i++) {
    pe_regs_setenv_nocopy(pe_regs, i - 3, args[i]);
  }
  for (i = 0; i < n; i++) {
    pe_regs_setenv_nocopy(pe_regs, 0, list[i]);
    if (call_ufun(&ufun, result, executor, enactor, pe_info, pe_regs))
      break;
    if ((check_bool == 0) ? (*result == '1' && *(result + 1) == '\0')
                          : parse_boolean(result)) {
      if (first)
        first = 0;
      else
        safe_str(osep, buff, bp);
      safe_str(list[i], buff, bp);
    }
    /* Can't do *bp == oldbp like in all the others, because bp might not
     * move even when not full, if one of the list elements is null and
     * we have a null separator. */
    if (*bp == (buff + BUFFER_LEN - 1) && pe_info->fun_invocations == funccount)
      break;
    funccount = pe_info->fun_invocations;
  }
  pe_regs_free(pe_regs);
  freearr(list, n);
}

/* ARGSUSED */
FUNCTION(fun_shuffle)
{
  /* given a list of words, randomize the order of words.
   * We do this by taking each element, and swapping it with another
   * element with a greater array index (thus, words[0] can be swapped
   * with anything up to words[n], words[5] with anything between
   * itself and words[n], etc.
   * This is relatively fast - linear time - and reasonably random.
   * Will take an optional delimiter argument.
   */

  char *words[MAX_SORTSIZE];
  int n, i, j;
  char sep;
  char *osep, osepd[2] = {'\0', '\0'};

  if (!delim_check(buff, bp, nargs, args, 2, &sep))
    return;

  if (nargs == 3)
    osep = args[2];
  else {
    osepd[0] = sep;
    osep = osepd;
  }

  /* split the list up, or return if the list is empty */
  if (!*args[0])
    return;
  n = list2arr_ansi(words, MAX_SORTSIZE, args[0], sep, 1);

  /* shuffle it */
  for (i = 0; i < n; i++) {
    char *tmp;
    j = get_random_u32(i, n - 1);
    tmp = words[j];
    words[j] = words[i];
    words[i] = tmp;
  }

  arr2list(words, n, buff, bp, osep);
  freearr(words, n);
}

/* ARGSUSED */
FUNCTION(fun_sort)
{
  char *ptrs[MAX_SORTSIZE];
  int nptrs;
  SortType sort_type;
  char sep;
  char *osep, osepd[2] = {'\0', '\0'};

  if (!nargs || !*args[0])
    return;

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;

  if (nargs < 4) {
    osepd[0] = sep;
    osep = osepd;
  } else
    osep = args[3];

  nptrs = list2arr_ansi(ptrs, MAX_SORTSIZE, args[0], sep, 1);
  sort_type = get_list_type(args, nargs, 2, ptrs, nptrs);
  do_gensort(executor, ptrs, NULL, nptrs, sort_type);
  arr2list(ptrs, nptrs, buff, bp, osep);
  freearr(ptrs, nptrs);
}

/* ARGSUSED */
FUNCTION(fun_sortkey)
{
  char *ptrs[MAX_SORTSIZE];
  char *keys[MAX_SORTSIZE];
  int nptrs;
  SortType sort_type;
  PE_REGS *pe_regs;
  char sep;
  char *osep, osepd[2] = {'\0', '\0'};
  int i;
  char result[BUFFER_LEN];
  ufun_attrib ufun;

  /* sortkey(attr,list,sort_type,delim,osep) */

  if (!nargs || !*args[0] || !*args[1])
    return;

  if (!delim_check(buff, bp, nargs, args, 4, &sep))
    return;

  if (nargs < 5) {
    osepd[0] = sep;
    osep = osepd;
  } else
    osep = args[4];

  /* find our object and attribute */
  if (!fetch_ufun_attrib(args[0], executor, &ufun, UFUN_DEFAULT))
    return;

  nptrs = list2arr_ansi(ptrs, MAX_SORTSIZE, args[1], sep, 1);

  pe_regs = pe_regs_create(PE_REGS_ARG, "fun_sortkey");

  /* Now we make a list of keys */
  for (i = 0; i < nptrs; i++) {
    /* Build our %0 args */
    pe_regs_setenv_nocopy(pe_regs, 0, ptrs[i]);
    call_ufun(&ufun, result, executor, enactor, pe_info, pe_regs);
    keys[i] = mush_strdup(result, "sortkey");
  }
  pe_regs_free(pe_regs);

  sort_type = get_list_type(args, nargs, 3, keys, nptrs);
  do_gensort(executor, keys, ptrs, nptrs, sort_type);
  arr2list(ptrs, nptrs, buff, bp, osep);
  freearr(ptrs, nptrs);
  for (i = 0; i < nptrs; i++) {
    mush_free(keys[i], "sortkey");
  }
}

/* ARGSUSED */
FUNCTION(fun_sortby)
{
  char *ptrs[MAX_SORTSIZE];
  char sep;
  int nptrs;
  char *osep, osepd[2] = {'\0', '\0'};
  ufun_attrib ufun;

  if (!nargs || !*args[0])
    return;

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;

  if (nargs == 4)
    osep = args[3];
  else {
    osepd[0] = sep;
    osep = osepd;
  }

  if (!fetch_ufun_attrib(args[0], executor, &ufun,
                         UFUN_DEFAULT | UFUN_REQUIRE_ATTR))
    return;

  /* Split up the list, sort it, reconstruct it. */
  nptrs = list2arr_ansi(ptrs, MAX_SORTSIZE, args[1], sep, 1);
  if (nptrs > 1) /* pointless to sort less than 2 elements */
    sane_qsort((void **) ptrs, 0, nptrs - 1, u_comp, executor, enactor, &ufun,
               pe_info);

  arr2list(ptrs, nptrs, buff, bp, osep);
  freearr(ptrs, nptrs);
}

#define OUTSEP()                                                               \
  do {                                                                         \
    if (found) {                                                               \
      safe_strl(osep, osepl, buff, bp);                                        \
    } else {                                                                   \
      found = 1;                                                               \
    }                                                                          \
  } while (0)

/* ARGSUSED */
FUNCTION(fun_setmanip)
{
  char sep;
  char **a1, **a2;
  int n1, n2, x1, x2, val;
  int orign1, orign2;
  int found = 0;
  SortType sort_type = UNKNOWN_LIST;
  int osepl = 0;
  char *osep = NULL, osepd[2] = {'\0', '\0'};
  s_rec *sp1, *sp2;
  ListTypeInfo *lti;

  bool dolt = 0, dogt = 0, doeq = 0;
  if (strstr(called_as, "SYMDIFF")) {
    dogt = dolt = 1;
  } else if (strstr(called_as, "DIFF")) {
    dolt = 1;
  } else if (strstr(called_as, "INTER")) {
    doeq = 1;
  } else {
    /* Setunion. */
    dolt = dogt = doeq = 1;
  }

  /* if no lists, then no work */
  if (!*args[0] && !*args[1])
    return;

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;

  a1 = mush_calloc(MAX_SORTSIZE, sizeof(char *), "ptrarray");
  a2 = mush_calloc(MAX_SORTSIZE, sizeof(char *), "ptrarray");
  if (!a1 || !a2)
    mush_panic("Unable to allocate memory in fun_setmanip");

  /* make arrays out of the lists */
  orign1 = n1 = list2arr_ansi(a1, MAX_SORTSIZE, args[0], sep, 1);
  orign2 = n2 = list2arr_ansi(a2, MAX_SORTSIZE, args[1], sep, 1);

  if (nargs < 4) {
    sort_type = autodetect_2lists(a1, n1, a2, n2);
    osepd[0] = sep;
    osep = osepd;
    if (sep)
      osepl = 1;
  } else if (nargs == 4) {
    sort_type = get_list_type_noauto(args, nargs, 4);
    if (sort_type == UNKNOWN_LIST) {
      osep = args[3];
      osepl = arglens[3];
    } else {
      osepd[0] = sep;
      osep = osepd;
      if (sep)
        osepl = 1;
    }
  } else if (nargs == 5) {
    sort_type = get_list_type_noauto(args, nargs, 4);
    osep = args[4];
    osepl = arglens[4];
  }

  if (sort_type == UNKNOWN_LIST) {
    sort_type = autodetect_2lists(a1, n1, a2, n2);
  }

  /* sort each array */
  lti = get_list_type_info(sort_type);
  sp1 = slist_build(executor, a1, NULL, n1, lti);
  sp2 = slist_build(executor, a2, NULL, n2, lti);
  slist_qsort(sp1, n1, lti);
  n1 = slist_uniq(sp1, n1, lti);
  slist_qsort(sp2, n2, lti);
  n2 = slist_uniq(sp2, n2, lti);

  /* get the first value for the intersection, removing duplicates
   * We already know that duplicates have been removed from each list. */
  x1 = x2 = 0;
  while ((x1 < n1) && (x2 < n2)) {
    val = slist_comp(&sp1[x1], &sp2[x2], lti);
    if (val < 0) {
      if (dolt) {
        OUTSEP();
        safe_str(sp1[x1].val, buff, bp);
      }
      x1++;
    } else if (val > 0) {
      if (dogt) {
        OUTSEP();
        safe_str(sp2[x2].val, buff, bp);
      }
      x2++;
    } else {
      if (doeq) {
        OUTSEP();
        safe_str(sp1[x1].val, buff, bp);
      }
      x1++;
      x2++;
    }
  }
  if (dolt) {
    while (x1 < n1) {
      OUTSEP();
      safe_str(sp1[x1++].val, buff, bp);
    }
  }
  if (dogt) {
    while (x2 < n2) {
      OUTSEP();
      safe_str(sp2[x2++].val, buff, bp);
    }
  }

  /* Free everything we've allocated. */
  slist_free(sp1, n1, lti);
  slist_free(sp2, n2, lti);
  free_list_type_info(lti);
  freearr(a1, orign1);
  freearr(a2, orign2);
  mush_free(a1, "ptrarray");
  mush_free(a2, "ptrarray");
}

FUNCTION(fun_unique)
{
  char sep;
  char **ary;
  int orign, n, i;
  int osepl = 0;
  char *osep = NULL, osepd[2] = {'\0', '\0'};
  SortType sort_type = ALPHANUM_LIST;
  ListTypeInfo *lti;
  s_rec *sp;

  /* if no lists, then no work */
  if (!*args[0])
    return;

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;

  ary = mush_calloc(MAX_SORTSIZE, sizeof(char *), "ptrarray");

  if (!ary)
    mush_panic("Unable to allocate memory in fun_unique");

  /* make an array out of the list */
  orign = n = list2arr_ansi(ary, MAX_SORTSIZE, args[0], sep, 1);

  if (nargs >= 2)
    sort_type = get_list_type(args, nargs, 2, ary, n);

  if (nargs < 4) {
    osepd[0] = sep;
    osep = osepd;
    if (sep)
      osepl = 1;
  } else if (nargs == 4) {
    osep = args[3];
    osepl = arglens[3];
  }

  lti = get_list_type_info(sort_type);
  sp = slist_build(executor, ary, NULL, n, lti);
  n = slist_uniq(sp, n, lti);

  for (i = 0; i < n; i++) {
    if (i > 0)
      safe_strl(osep, osepl, buff, bp);
    safe_str(sp[i].val, buff, bp);
  }

  slist_free(sp, n, lti);
  free_list_type_info(lti);
  freearr(ary, orign);
  mush_free(ary, "ptrarray");
}

#define CACHE_SIZE 8 /**< Maximum size of the lnum cache */

/* ARGSUSED */
FUNCTION(fun_lnum)
{
  NVAL j;
  NVAL start;
  NVAL end;
  NVAL step = 1.0;
  int istart, iend, k, istep;
  char const *osep = " ";
  static NVAL cstart[CACHE_SIZE];
  static NVAL cend[CACHE_SIZE];
  static char csep[CACHE_SIZE][BUFFER_LEN];
  static char cresult[CACHE_SIZE][BUFFER_LEN];
  static int cstep[CACHE_SIZE];
  static int cpos;
  char *cp;

  if (!is_number(args[0])) {
    safe_str(T(e_num), buff, bp);
    return;
  }
  end = parse_number(args[0]);
  if (nargs > 1) {
    if (!is_number(args[1])) {
      safe_str(T(e_num), buff, bp);
      return;
    }
    if (nargs > 3 && is_number(args[3])) {
      step = parse_number(args[3]);
    }
    start = end;
    end = parse_number(args[1]);
    if ((start == 0) && (end == 0)) {
      safe_str("0", buff, bp); /* Special case - lnum(0,0) -> 0 */
      return;
    }
  } else {
    if (end == 0.0)
      return; /* Special case - lnum(0) -> blank string */
    else if (end == 1.0) {
      safe_str("0", buff, bp); /* Special case - lnum(1) -> 0 */
      return;
    }
    end--;
    if (end < 0.0) {
      safe_str(T("#-1 NUMBER OUT OF RANGE"), buff, bp);
      return;
    }
    start = 0.0;
  }
  if (nargs > 2) {
    osep = args[2];
  }
  for (k = 0; k < CACHE_SIZE; k++) {
    if (cstart[k] == start && cend[k] == end && !strcmp(csep[k], osep) &&
        cstep[k] == step) {
      safe_str(cresult[k], buff, bp);
      return;
    }
  }
  cpos = (cpos + 1) % CACHE_SIZE;
  cstart[cpos] = start;
  cend[cpos] = end;
  strcpy(csep[cpos], osep);
  cp = cresult[cpos];

  if (step == 0.0)
    step = 1.0;
  else if (step < 0.0)
    step = step * -1.0;

  istart = (int) start;
  iend = (int) end;
  istep = (int) step;
  if (istart == start && iend == end && istep == step) {
    safe_integer(istart, cresult[cpos], &cp);
    if (istart <= iend) {
      for (k = istart + istep; k <= iend; k += istep) {
        safe_str(osep, cresult[cpos], &cp);
        if (safe_integer(k, cresult[cpos], &cp))
          break;
      }
    } else {
      for (k = istart - istep; k >= iend; k -= istep) {
        safe_str(osep, cresult[cpos], &cp);
        if (safe_integer(k, cresult[cpos], &cp))
          break;
      }
    }
  } else {
    safe_number(start, cresult[cpos], &cp);
    if (start <= end) {
      for (j = start + step; j <= end; j += step) {
        safe_str(osep, cresult[cpos], &cp);
        if (safe_number(j, cresult[cpos], &cp))
          break;
      }
    } else {
      for (j = start - step; j >= end; j -= step) {
        safe_str(osep, cresult[cpos], &cp);
        if (safe_number(j, cresult[cpos], &cp))
          break;
      }
    }
  }
  *cp = '\0';

  safe_str(cresult[cpos], buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_first)
{
  /* read first word from a string */

  char *p;
  char sep;

  if (!*args[0])
    return;

  if (!delim_check(buff, bp, nargs, args, 2, &sep))
    return;

  if (has_markup(args[0])) {
    ansi_string *as = NULL;
    char *q;
    as = parse_ansi_string(args[0]);
    p = trim_space_sep(as->text, sep);
    q = split_token(&p, sep);
    if (p)
      safe_ansi_string(as, q - as->text, p - q - 1, buff, bp);
    else
      safe_ansi_string(as, q - as->text, as->len, buff, bp);
    free_ansi_string(as);
  } else {
    p = trim_space_sep(args[0], sep);
    safe_str(split_token(&p, sep), buff, bp);
  }
}

enum rand_types { RAND_LINEAR, RAND_NODUPLICATES, RAND_DUPLICATES };
/* ARGSUSED */
FUNCTION(fun_randword)
{
  char **ptrs;
  int nptrs;
  char sep;
  char *osep;
  char osepd[2] = {'\0', '\0'};
  int separg = 2;
  int randcount = 1;
  enum rand_types randtype = RAND_NODUPLICATES;
  bool first = 1;
  int word_index = -1;

  if (!*args[0]) {
    return;
  }
  if (!strcmp(called_as, "RANDEXTRACT")) {
    separg = 3;
    if (nargs > 1) {
      if (args[1] && *args[1] && !is_strict_integer(args[1])) {
        safe_str(T(e_int), buff, bp);
        return;
      }
      randcount = parse_integer(args[1]);
      if (randcount < 1) {
        return;
      }
    }
  }

  if (!delim_check(buff, bp, nargs, args, separg, &sep)) {
    return;
  }

  if (nargs > 3 && args[3] && *args[3]) {
    if (*args[3] == 'l' || *args[3] == 'L') {
      randtype = RAND_LINEAR;
    } else if (*args[3] == 'r' || *args[3] == 'R') {
      randtype = RAND_NODUPLICATES;
    } else if (*args[3] == 'd' || *args[3] == 'D') {
      randtype = RAND_DUPLICATES;
    } else {
      safe_str(T("#-1 INVALID TYPE"), buff, bp);
      return;
    }
  }

  if (nargs > 4) {
    osep = args[4];
  } else {
    osepd[0] = sep;
    osep = osepd;
  }

  ptrs = mush_calloc(MAX_SORTSIZE, sizeof(char *), "ptrarray");
  nptrs = list2arr_ansi(ptrs, MAX_SORTSIZE, args[0], sep, 1);

  if (!nptrs) {
    return;
  }
  if (randcount > nptrs && randtype != RAND_DUPLICATES) {
    randcount = nptrs;
  }

  for (; randcount; randcount--) {
    if (word_index == -1 || randtype != RAND_LINEAR) {
      word_index = get_random_u32(0, nptrs - 1);
    } else {
      word_index++;
      if (word_index >= nptrs) {
        break; /* End of string */
      }
    }
    while (!ptrs[word_index]) {
      /* Find an unused word - there will always be one */
      word_index++;
      if (word_index > nptrs) {
        word_index = 0; /* Back to beginning */
      }
    }
    if (first) {
      first = 0;
    } else {
      safe_str(osep, buff, bp);
    }
    safe_str(ptrs[word_index], buff, bp);
    if (randtype == RAND_NODUPLICATES) {
      /* Don't pick this word again */
      freearr_member(ptrs[word_index]);
      ptrs[word_index] = NULL;
    }
  }
  freearr(ptrs, nptrs);
  mush_free(ptrs, "ptrarray");
}

/* ARGSUSED */
FUNCTION(fun_rest)
{
  char *p;
  char sep;
  ansi_string *as = NULL;

  if (!*args[0])
    return;

  if (!delim_check(buff, bp, nargs, args, 2, &sep))
    return;

  p = trim_space_sep(args[0], sep);
  if (has_markup(p)) {
    as = parse_ansi_string(p);
    p = as->text;
  }
  (void) split_token(&p, sep);
  if (!p || !*p)
    return;
  if (as) {
    safe_ansi_string(as, p - as->text, as->len, buff, bp);
    free_ansi_string(as);
  } else {
    safe_str(p, buff, bp);
  }
}

/* ARGSUSED */
FUNCTION(fun_last)
{
  /* read last word from a string */

  char *p, *r;
  char sep;
  ansi_string *as = NULL;

  if (!*args[0])
    return;

  if (!delim_check(buff, bp, nargs, args, 2, &sep))
    return;

  p = trim_space_sep(args[0], sep);

  if (has_markup(p)) {
    as = parse_ansi_string(p);
    p = as->text;
  }

  if (!(r = strrchr(p, sep))) {
    if (as) {
      safe_ansi_string(as, 0, as->len, buff, bp);
    } else {
      safe_str(p, buff, bp);
    }
  } else {
    r++;
    if (as) {
      safe_ansi_string(as, r - p, as->len, buff, bp);
    } else {
      safe_str(r, buff, bp);
    }
  }
  if (as) {
    free_ansi_string(as);
  }
}

/* ARGSUSED */
FUNCTION(fun_grab)
{
  char *r, *s, sep;
  ansi_string *as = NULL;

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;

  if (has_markup(args[0])) {
    as = parse_ansi_string(args[0]);
    s = trim_space_sep(as->text, sep);
  } else {
    s = trim_space_sep(args[0], sep);
  }

  do {
    r = split_token(&s, sep);
    if (r && quick_wild(args[1], r)) {
      if (as) {
        safe_ansi_string(as, r - as->text, s - r - 1, buff, bp);
        free_ansi_string(as);
      } else {
        safe_str(r, buff, bp);
      }
      return;
    }
  } while (s);
  if (as)
    free_ansi_string(as);
}

/* ARGSUSED */
FUNCTION(fun_namegraball)
{
  /* Given a list of dbrefs and a string, it matches the
   * name of the dbrefs against the string.
   * grabnameall(#1 #2 #3,god) -> #1
   */

  char *r, *s, sep;
  dbref victim;
  dbref absolute;
  int first = 1;

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;

  absolute = parse_objid(args[1]);
  if (!RealGoodObject(absolute))
    absolute = NOTHING;

  if (*args[1]) {
    s = trim_space_sep(args[0], sep);
    do {
      r = split_token(&s, sep);
      victim = parse_objid(r);
      if (!RealGoodObject(victim))
        continue; /* Don't bother with garbage */
      if (!(string_match(Name(victim), args[1]) || (absolute == victim)))
        continue;
      if (!can_interact(victim, executor, INTERACT_MATCH, pe_info))
        continue;
      /* It matches, and is interact-able */
      if (!first)
        safe_chr(sep, buff, bp);
      safe_str(r, buff, bp);
      first = 0;
    } while (s);
  } else {
    /* Pull out all good objects (those that _have_ names) */
    s = trim_space_sep(args[0], sep);
    do {
      r = split_token(&s, sep);
      victim = parse_objid(r);
      if (!RealGoodObject(victim))
        continue; /* Don't bother with garbage */
      if (!can_interact(victim, executor, INTERACT_MATCH, pe_info))
        continue;
      /* It's real, and is interact-able */
      if (!first)
        safe_chr(sep, buff, bp);
      safe_str(r, buff, bp);
      first = 0;
    } while (s);
  }
}

/* ARGSUSED */
FUNCTION(fun_namegrab)
{
  /* Given a list of dbrefs and a string, it matches the
   * name of the dbrefs against the string.
   */

  char *r, *s, sep;
  dbref victim;
  dbref absolute;
  char *exact_res, *res;

  exact_res = res = NULL;

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;

  absolute = parse_objid(args[1]);
  if (!RealGoodObject(absolute))
    absolute = NOTHING;

  /* Walk the wordstring, until we find the word we want. */
  s = trim_space_sep(args[0], sep);
  do {
    r = split_token(&s, sep);
    victim = parse_objid(r);
    if (!RealGoodObject(victim))
      continue; /* Don't bother with garbage */
    /* Dbref match has top priority */
    if ((absolute == victim) &&
        can_interact(victim, executor, INTERACT_MATCH, pe_info)) {
      safe_str(r, buff, bp);
      return;
    }
    /* Exact match has second priority */
    if (!exact_res && !strcasecmp(Name(victim), args[1]) &&
        can_interact(victim, executor, INTERACT_MATCH, pe_info)) {
      exact_res = r;
    }
    /* Non-exact match. */
    if (!res && string_match(Name(victim), args[1]) &&
        can_interact(victim, executor, INTERACT_MATCH, pe_info)) {
      res = r;
    }
  } while (s);
  if (exact_res)
    safe_str(exact_res, buff, bp);
  else if (res)
    safe_str(res, buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_match)
{
  /* compares two strings with possible wildcards, returns the
   * word position of the match. Based on the 2.0 version of this
   * function.
   */

  char *s, *r;
  char sep;
  int wcount = 1;

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;

  /* Walk the wordstring, until we find the word we want.
   * The ANSI in args[1] isn't properly preserved, but since we aren't
   * returning the string anyway, we don't care
   */
  s = trim_space_sep(args[0], sep);
  do {
    r = split_token(&s, sep);
    if (quick_wild(args[1], r)) {
      safe_integer(wcount, buff, bp);
      return;
    }
    wcount++;
  } while (s);
  safe_chr('0', buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_wordpos)
{
  int charpos, i;
  char *cp, *tp, *xp;
  char sep;

  if (!is_integer(args[1])) {
    safe_str(T(e_int), buff, bp);
    return;
  }
  charpos = parse_integer(args[1]);
  cp = args[0];
  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;

  if ((charpos <= 0) || ((size_t) charpos > strlen(cp))) {
    safe_str("#-1", buff, bp);
    return;
  }
  tp = cp + charpos - 1;
  cp = trim_space_sep(cp, sep);
  xp = split_token(&cp, sep);
  for (i = 1; xp; i++) {
    if (tp < (xp + strlen(xp)))
      break;
    xp = split_token(&cp, sep);
  }
  safe_integer(i, buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_extract)
{
  int nwords;
  char **ptrs;
  char *wordlist;
  char sep;
  int start = 0, len = 1, first = 1;

  if (!delim_check(buff, bp, nargs, args, 4, &sep))
    return;

  ptrs = mush_calloc(MAX_SORTSIZE, sizeof(char *), "ptrarray");
  wordlist = mush_malloc(BUFFER_LEN, "string");
  if (!ptrs)
    mush_panic("Unable to allocate memory in fun_extract");

  /* Turn the first list into an array. */
  strcpy(wordlist, args[0]);

  if (nargs > 1) {
    /* find_list_position does an is_integer check, but we
     * duplicate it here so we can return e_ints */
    if (!is_integer(args[1])) {
      safe_str(T(e_ints), buff, bp);
      return;
    }
    if (nargs > 2 && !is_integer(args[2])) {
      safe_str(T(e_ints), buff, bp);
      return;
    }
  }

  nwords = list2arr_ansi(ptrs, MAX_SORTSIZE, wordlist, sep, 1);

  if (nargs > 1)
    start = find_list_position(args[1], nwords, 0) - 1;

  if (nargs > 2) {
    len = parse_integer(args[2]);
    if (len < 0)
      len = find_list_position(args[2], nwords, 0) - start;
  }

  if (start < 0 || start >= nwords || len < 1) {
    freearr(ptrs, nwords);
    mush_free(ptrs, "ptrarray");
    mush_free(wordlist, "string");
    return;
  }

  for (; start < nwords && len; start++, len--) {
    if (first)
      first = 0;
    else
      safe_chr(sep, buff, bp);
    safe_str(ptrs[start], buff, bp);
  }

  freearr(ptrs, nwords);
  mush_free(ptrs, "ptrarray");
  mush_free(wordlist, "string");
}

/* ARGSUSED */
FUNCTION(fun_cat)
{
  int i;

  safe_strl(args[0], arglens[0], buff, bp);
  for (i = 1; i < nargs; i++) {
    safe_chr(' ', buff, bp);
    safe_strl(args[i], arglens[i], buff, bp);
  }
}

/* ARGSUSED */
FUNCTION(fun_remove)
{
  char sep;
  char **list, **rem;
  int list_counter, rem_counter, list_total, rem_total;
  int skip[MAX_SORTSIZE] = {0};
  int first = 1;

  /* zap word from string */

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;

  list = mush_calloc(MAX_SORTSIZE, sizeof(char *), "ptrarray");
  rem = mush_calloc(MAX_SORTSIZE, sizeof(char *), "ptrarray");

  list_total = list2arr_ansi(list, MAX_SORTSIZE, args[0], sep, 1);
  rem_total = list2arr_ansi(rem, MAX_SORTSIZE, args[1], sep, 1);

  for (rem_counter = 0; rem_counter < rem_total; rem_counter++) {
    for (list_counter = 0; list_counter < list_total; list_counter++) {
      if (!skip[list_counter] &&
          !ansi_strcmp(rem[rem_counter], list[list_counter])) {
        skip[list_counter] = 1;
        break;
      }
    }
  }

  for (list_counter = 0; list_counter < list_total; list_counter++) {
    if (skip[list_counter])
      continue;
    if (first)
      first = 0;
    else
      safe_chr(sep, buff, bp);
    safe_str(list[list_counter], buff, bp);
  }

  freearr(list, list_total);
  freearr(rem, rem_total);
  mush_free(list, "ptrarray");
  mush_free(rem, "ptrarray");
}

/* ARGSUSED */
FUNCTION(fun_items)
{
  /* the equivalent of WORDS for an arbitrary separator */
  /* This differs from WORDS in its treatment of the space
   * separator.
   */

  char *s = args[0];
  char c = *args[1];
  int count = 1;

  if (c == '\0')
    c = ' ';

  while ((s = strchr(s, c))) {
    count++;
    s++;
  }

  safe_integer(count, buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_index)
{
  /* more or less the equivalent of EXTRACT for an arbitrary separator */
  /* This differs from EXTRACT in its handling of space separators. */

  int start, end;
  char c;
  char *s, *p;

  if (!is_integer(args[2]) || !is_integer(args[3])) {
    safe_str(T(e_ints), buff, bp);
    return;
  }
  s = args[0];
  c = *args[1];
  if (!c)
    c = ' ';

  start = parse_integer(args[2]);
  end = parse_integer(args[3]);

  if ((start < 1) || (end < 1) || (*s == '\0'))
    return;

  /* move s to the start of the item we want */
  while (--start) {
    if (!(s = strchr(s, c)))
      return;
    s++;
  }

  /* skip just spaces, not tabs or newlines, since people may MUSHcode things
   * like "%r%tPolgara %r%tDurnik %r%tJavelin"
   */
  while (*s == ' ')
    s++;
  if (!*s)
    return;

  /* now figure out where to end the string */
  p = s + 1;
  /* we may already be pointing to a separator */
  if (*s == c)
    end--;
  while (end--)
    if (!(p = strchr(p, c)))
      break;
    else
      p++;

  if (p)
    p--;
  else
    p = s + strlen(s);

  /* trim trailing spaces (just true spaces) */
  while ((p > s) && (p[-1] == ' '))
    p--;
  *p = '\0';

  safe_str(s, buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_ldelete)
{
  /* delete or replace a word at given positions of a list */
  /* This code powers LDELETE() and REPLACE() */

  /* Given a list and a list of numbers, delete the corresponding
   * elements of the list. ldelete(ack bar eep foof yay,2 4) = ack eep yay
   * A separator for the first list is allowed.
   * This code modified slightly from 'elements'
   */
  int nwords, cur;
  char **ptrs;
  char *wordlist;
  int first = 0;
  char *s, *r, sep;
  char *osep, osepd[2] = {'\0', '\0'};
  int delimarg = 3;
  char *replace = NULL;

  if (!strcmp(called_as, "LREPLACE")) {
    delimarg = 4;
    replace = args[2];
  }

  if (!delim_check(buff, bp, nargs, args, delimarg, &sep))
    return;

  if (nargs == (delimarg + 1))
    osep = args[delimarg];
  else {
    osepd[0] = sep;
    osep = osepd;
  }

  ptrs = mush_calloc(MAX_SORTSIZE, sizeof(char *), "ptrarray");
  wordlist = mush_malloc(BUFFER_LEN, "string");
  if (!ptrs || !wordlist)
    mush_panic("Unable to allocate memory in fun_ldelete");

  /* Turn the first list into an array. */
  strcpy(wordlist, args[0]);
  nwords = list2arr_ansi(ptrs, MAX_SORTSIZE, wordlist, sep, 1);

  s = trim_space_sep(args[1], ' ');

  /* Go through the second list, grabbing the numbers and finding the
   * corresponding elements.
   */
  do {
    r = split_token(&s, ' ');
    cur = find_list_position(r, nwords, 0) - 1;
    if ((cur >= 0) && (cur < nwords)) {
      if (replace) {
        free(ptrs[cur]);
        ptrs[cur] = strdup(replace);
      } else {
        freearr_member(ptrs[cur]);
        ptrs[cur] = NULL;
      }
    }
  } while (s);
  for (cur = 0; cur < nwords; cur++) {
    if (ptrs[cur]) {
      if (first)
        safe_str(osep, buff, bp);
      else
        first = 1;
      safe_str(ptrs[cur], buff, bp);
    }
  }

  freearr(ptrs, nwords);
  mush_free(ptrs, "ptrarray");
  mush_free(wordlist, "string");
}

/* ARGSUSED */
FUNCTION(fun_insert)
{
  /* insert a word at position X of a list */
  int nwords, pos, i;
  char **ptrs;
  char *wordlist;
  int first = 1;
  char sep;

  if (!delim_check(buff, bp, nargs, args, 4, &sep))
    return;

  ptrs = mush_calloc(MAX_SORTSIZE, sizeof(char *), "ptrarray");
  wordlist = mush_malloc(BUFFER_LEN, "string");
  if (!ptrs || !wordlist)
    mush_panic("Unable to allocate memory in fun_insert");

  /* Turn the first list into an array. */
  strcpy(wordlist, args[0]);
  nwords = list2arr_ansi(ptrs, MAX_SORTSIZE, wordlist, sep, 1);

  /* Go through the second list, grabbing the numbers and finding the
   * corresponding elements.
   */
  pos = find_list_position(args[1], nwords, 1) - 1;
  if (pos < 0 || pos > (nwords + 1)) {
    /* Nowhere to insert */
    safe_strl(args[0], arglens[0], buff, bp);
  } else {
    for (i = 0; i < nwords; i++) {
      if (i == pos) {
        if (first)
          first = 0;
        else
          safe_chr(sep, buff, bp);
        safe_strl(args[2], arglens[2], buff, bp);
      }
      if (first)
        first = 0;
      else
        safe_chr(sep, buff, bp);
      safe_str(ptrs[i], buff, bp);
    }
    if (i == pos) {
      /* When pos was given as -1, we might be inserting after the list */
      if (!first)
        safe_chr(sep, buff, bp);
      safe_strl(args[2], arglens[2], buff, bp);
    }
  }

  freearr(ptrs, nwords);
  mush_free(ptrs, "ptrarray");
  mush_free(wordlist, "string");
}

/* ARGSUSED */
FUNCTION(fun_member)
{
  char *s, *t;
  char sep;
  int el;

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;

  if (strchr(args[1], sep)) {
    safe_str(T("#-1 CAN ONLY TEST ONE ELEMENT"), buff, bp);
    return;
  }

  s = trim_space_sep(args[0], sep);
  el = 1;

  do {
    t = split_token(&s, sep);
    if (!strcmp(args[1], t)) {
      safe_integer(el, buff, bp);
      return;
    }
    el++;
  } while (s);

  safe_chr('0', buff, bp); /* not found */
}

/* ARGSUSED */
FUNCTION(fun_before)
{
  const char *p, *q;
  ansi_string *as;
  size_t len;
  p = remove_markup(args[1], &len);

  if (!*p)
    p = " ";
  as = parse_ansi_string(args[0]);
  q = strstr(as->text, p);
  if (q) {
    safe_ansi_string(as, 0, q - as->text, buff, bp);
  } else {
    safe_strl(args[0], arglens[0], buff, bp);
  }
  free_ansi_string(as);
}

/* ARGSUSED */
FUNCTION(fun_after)
{
  ansi_string *as;
  char *p, *delim;
  size_t len, count;
  size_t start;

  if (!*args[1]) {
    args[1][0] = ' ';
    args[1][1] = '\0';
    arglens[1] = 1;
  }
  delim = remove_markup(args[1], &len);
  len--;
  as = parse_ansi_string(args[0]);

  p = strstr(as->text, delim);
  if (p) {
    start = p - as->text + len;
    count = as->len - start;
    safe_ansi_string(as, start, count, buff, bp);
  }
  free_ansi_string(as);
}

/* ARGSUSED */
FUNCTION(fun_revwords)
{
  char **words;
  int count, origcount;
  char sep;
  char *osep, osepd[2] = {'\0', '\0'};

  if (!delim_check(buff, bp, nargs, args, 2, &sep))
    return;

  if (nargs == 3)
    osep = args[2];
  else {
    osepd[0] = sep;
    osep = osepd;
  }

  words = mush_calloc(BUFFER_LEN, sizeof(char *), "wordlist");

  origcount = count = list2arr_ansi(words, BUFFER_LEN, args[0], sep, 1);
  if (count == 0) {
    mush_free(words, "wordlist");
    return;
  }

  safe_str(words[--count], buff, bp);
  while (count) {
    safe_str(osep, buff, bp);
    safe_str(words[--count], buff, bp);
  }
  freearr(words, origcount);
  mush_free(words, "wordlist");
}

/* ARGSUSED */
FUNCTION(fun_words)
{
  char sep;

  if (!delim_check(buff, bp, nargs, args, 2, &sep))
    return;
  safe_integer(do_wordcount(trim_space_sep(args[0], sep), sep), buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_splice)
{
  /* like MERGE(), but does it for a word */
  char *orig[MAX_SORTSIZE];
  char *repl[MAX_SORTSIZE];
  char haystack[BUFFER_LEN];
  int ocount, rcount;
  int i;
  char sep;
  char osep[2];

  if (!delim_check(buff, bp, nargs, args, 4, &sep))
    return;

  osep[0] = sep;
  osep[1] = '\0';

  mush_strncpy(haystack, remove_markup(args[2], NULL), sizeof haystack);
  if (!*haystack) {
    safe_str(T("#-1 NEED A WORD"), buff, bp);
    return;
  }
  if (do_wordcount(haystack, sep) != 1) {
    safe_str(T("#-1 TOO MANY WORDS"), buff, bp);
    return;
  }

  /* Turn them into lists */
  ocount = list2arr_ansi(orig, MAX_SORTSIZE, args[0], sep, 1);
  rcount = list2arr_ansi(repl, MAX_SORTSIZE, args[1], sep, 1);

  if (ocount != rcount) {
    safe_str(T("#-1 NUMBER OF WORDS MUST BE EQUAL"), buff, bp);
    freearr(orig, ocount);
    freearr(repl, rcount);
    return;
  }

  for (i = 0; i < ocount; i++) {
    if (!strcmp(remove_markup(orig[i], NULL), haystack)) {
      char *tmp = orig[i];
      orig[i] = repl[i];
      repl[i] = tmp;
    }
  }

  arr2list(orig, ocount, buff, bp, osep);

  freearr(orig, ocount);
  freearr(repl, rcount);
}

/* ARGSUSED */
FUNCTION(fun_iter)
{
  /* Based on the TinyMUSH 2.0 code for this function. Please note that
   * arguments to this function are passed _unparsed_.
   */
  /* Actually, this code has changed so much that the above comment
   * isn't really true anymore. - Talek, 18 Oct 2000
   */
  char **ptrs = NULL;
  int nptrs, i;

  char sep;
  char *outsep, *list;
  char *tbuf2, *lp;
  char const *sp;
  int funccount, per;
  const char *replace[2];
  PE_REGS *pe_regs;

  if (nargs >= 3) {
    /* We have a delimiter. We've got to parse the third arg in place */
    char insep[BUFFER_LEN];
    char *isep = insep;
    const char *arg3 = args[2];
    if (process_expression(insep, &isep, &arg3, executor, caller, enactor,
                           eflags, PT_DEFAULT, pe_info))
      return;
    *isep = '\0';
    strcpy(args[2], insep);
  }
  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;

  outsep = mush_malloc(BUFFER_LEN, "string");
  list = mush_malloc(BUFFER_LEN, "string");
  if (!outsep || !list) {
    mush_panic("Unable to allocate memory in fun_iter");
  }
  if (nargs < 4) {
    strcpy(outsep, " ");
  } else {
    const char *arg4 = args[3];
    char *osep = outsep;
    process_expression(outsep, &osep, &arg4, executor, caller, enactor, eflags,
                       PT_DEFAULT, pe_info);
    *osep = '\0';
  }
  lp = list;
  sp = args[0];
  per = process_expression(list, &lp, &sp, executor, caller, enactor, eflags,
                           PT_DEFAULT, pe_info);
  *lp = '\0';
  lp = trim_space_sep(list, sep);
  if (per || !*lp) {
    mush_free(outsep, "string");
    mush_free(list, "string");
    return;
  }

  /* Split lp up into an ansi-safe list */
  ptrs = mush_calloc(MAX_SORTSIZE, sizeof(char *), "ptrarray");
  nptrs = list2arr_ansi(ptrs, MAX_SORTSIZE, lp, sep, 1);

  funccount = pe_info->fun_invocations;

  pe_regs = pe_regs_localize(pe_info, PE_REGS_ITER, "fun_iter");
  for (i = 0; i < nptrs; i++) {
    if (i > 0) {
      safe_str(outsep, buff, bp);
    }
    pe_regs_set(pe_regs, PE_REGS_ITER, "t0", ptrs[i]);
    pe_regs_set_int(pe_regs, PE_REGS_ITER, "n0", i + 1);
    replace[0] = ptrs[i];
    replace[1] = unparse_integer(i + 1);

    tbuf2 = replace_string2(standard_tokens, replace, args[1]);
    sp = tbuf2;
    if (process_expression(buff, bp, &sp, executor, caller, enactor, eflags,
                           PT_DEFAULT, pe_info)) {
      mush_free(tbuf2, "replace_string.buff");
      break;
    }
    if (*bp == (buff + BUFFER_LEN - 1) &&
        pe_info->fun_invocations == funccount) {
      mush_free(tbuf2, "replace_string.buff");
      break;
    }
    funccount = pe_info->fun_invocations;
    mush_free(tbuf2, "replace_string.buff");
    if (pe_regs->flags & PE_REGS_IBREAK) {
      break;
    }
  }
  pe_regs_restore(pe_info, pe_regs);
  pe_regs_free(pe_regs);
  mush_free(outsep, "string");
  mush_free(list, "string");
  freearr(ptrs, nptrs);
  mush_free(ptrs, "ptrarray");
}

/* ARGSUSED */
FUNCTION(fun_ibreak)
{
  int i = 1;
  PE_REGS *pe_regs;
  int maxlev = PE_Get_Ilev(pe_info);

  /* maxlev is 0-based. ibreak is 1-based. Blah */
  maxlev += 1;

  if (nargs && args[0] && *args[0]) {
    if (!is_strict_integer(args[0])) {
      safe_str(T(e_int), buff, bp);
      return;
    }
    i = parse_integer(args[0]);
  }

  if (i == 0)
    return;

  if (i < 0 || i > maxlev) {
    safe_str(T(e_range), buff, bp);
    return;
  }

  for (pe_regs = pe_info->regvals; i > 0 && pe_regs; pe_regs = pe_regs->prev) {
    if (pe_regs->flags & PE_REGS_ITER) {
      pe_regs->flags |= PE_REGS_IBREAK;
      i--;
    }
    if (pe_regs->flags & PE_REGS_NEWATTR) {
      break;
    }
  }
}

/* ARGSUSED */
FUNCTION(fun_ilev) { safe_integer(PE_Get_Ilev(pe_info), buff, bp); }

/* ARGSUSED */
FUNCTION(fun_itext)
{
  int i;
  int maxlev = PE_Get_Ilev(pe_info);

  if (!strcasecmp(args[0], "l")) {
    i = maxlev;
  } else {
    if (!is_strict_integer(args[0])) {
      safe_str(T(e_int), buff, bp);
      return;
    }
    i = parse_integer(args[0]);
  }

  if (i < 0 || i > maxlev) {
    safe_str(T(e_argrange), buff, bp);
    return;
  }

  safe_str(PE_Get_Itext(pe_info, i), buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_inum)
{
  int i;
  int maxlev = PE_Get_Ilev(pe_info);

  if (!strcasecmp(args[0], "l")) {
    i = maxlev;
  } else {
    if (!is_strict_integer(args[0])) {
      safe_str(T(e_int), buff, bp);
      return;
    }
    i = parse_integer(args[0]);
  }

  if (i < 0 || i > maxlev) {
    safe_str(T(e_argrange), buff, bp);
    return;
  }

  safe_integer(PE_Get_Inum(pe_info, i), buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_step)
{
  /* Like map, but passes up to MAX_STACK_ARGS elements from the list at a time
   * in %0-%9
   * If the attribute is not found, null is returned, NOT an error.
   * This function takes delimiters.
   */

  char *lp;
  char sep;
  int n;
  int step;
  char *osep, osepd[2] = {'\0', '\0'};
  PE_REGS *pe_regs = NULL;
  ufun_attrib ufun;
  char rbuff[BUFFER_LEN];
  char **ptrs = NULL;
  int nptrs, i;

  if (!is_integer(args[2])) {
    safe_str(T(e_int), buff, bp);
    return;
  }

  step = parse_integer(args[2]);

  if (step < 1 || step > MAX_STACK_ARGS) {
    safe_str(T("#-1 STEP OUT OF RANGE"), buff, bp);
    return;
  }

  if (!delim_check(buff, bp, nargs, args, 4, &sep))
    return;

  if (nargs == 5)
    osep = args[4];
  else {
    osepd[0] = sep;
    osep = osepd;
  }

  lp = trim_space_sep(args[1], sep);
  if (!*lp)
    return;

  /* find our object and attribute */
  if (!fetch_ufun_attrib(args[0], executor, &ufun, UFUN_DEFAULT))
    return;

  /* Split lp up into an ansi-safe list */
  ptrs = mush_calloc(MAX_SORTSIZE, sizeof(char *), "ptrarray");
  nptrs = list2arr_ansi(ptrs, MAX_SORTSIZE, lp, sep, 1);

  /* Step through the list. */
  for (i = 0; i < nptrs;) {
    pe_regs = pe_regs_create(PE_REGS_ARG, "fun_step");
    for (n = 0; n < step; n++) {
      if (i < nptrs) {
        pe_regs_setenv_nocopy(pe_regs, n, ptrs[i++]);
      }
    }
    if (call_ufun(&ufun, rbuff, executor, enactor, pe_info, pe_regs)) {
      goto exitsequence;
    }
    pe_regs_free(pe_regs);
    pe_regs = NULL;
    if (i > step) {
      /* At least second loop */
      safe_str(osep, buff, bp);
    }
    safe_str(rbuff, buff, bp);
  }
exitsequence:
  if (pe_regs)
    pe_regs_free(pe_regs);
  freearr(ptrs, nptrs);
  mush_free(ptrs, "ptrarray");
}

/* ARGSUSED */
FUNCTION(fun_map)
{
  /* Like iter(), but calls an attribute with list elements as %0 instead.
   * If the attribute is not found, null is returned, NOT an error.
   * This function takes delimiters.
   */

  ufun_attrib ufun;
  char *lp;
  PE_REGS *pe_regs;
  char sep;
  int funccount;
  char place[16];
  char *osep, osepd[2] = {'\0', '\0'};
  char rbuff[BUFFER_LEN];
  char **ptrs = NULL;
  int nptrs, i;

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;

  osepd[0] = sep;
  osep = (nargs >= 4) ? args[3] : osepd;

  lp = trim_space_sep(args[1], sep);
  if (!*lp)
    return;

  if (!fetch_ufun_attrib(args[0], executor, &ufun, UFUN_DEFAULT))
    return;

  strcpy(place, "1");

  ptrs = mush_calloc(MAX_SORTSIZE, sizeof(char *), "ptrarray");
  nptrs = list2arr_ansi(ptrs, MAX_SORTSIZE, lp, sep, 1);

  /* Build our %0 args */
  pe_regs = pe_regs_create(PE_REGS_ARG, "fun_map");
  pe_regs_setenv_nocopy(pe_regs, 1, place);
  for (i = 0; i < nptrs; i++) {
    pe_regs_setenv_nocopy(pe_regs, 0, ptrs[i]);
    snprintf(place, 16, "%d", i + 1);

    funccount = pe_info->fun_invocations;

    if (call_ufun(&ufun, rbuff, executor, enactor, pe_info, pe_regs)) {
      break;
    }

    if (i > 0) {
      safe_str(osep, buff, bp);
    }
    safe_str(rbuff, buff, bp);
    if (*bp >= (buff + BUFFER_LEN - 1) &&
        pe_info->fun_invocations == funccount) {
      break;
    }
  }
  pe_regs_free(pe_regs);
  freearr(ptrs, nptrs);
  mush_free(ptrs, "ptrarray");
}

/* ARGSUSED */
FUNCTION(fun_mix)
{
  /* Like map(), but goes through lists, passing them as %0 and %1.. %9.
   * If the attribute is not found, null is returned, NOT an error.
   * This function takes delimiters.
   */

  ufun_attrib ufun;
  char rbuff[BUFFER_LEN];
  char *lp[MAX_STACK_ARGS];
  char sep;
  PE_REGS *pe_regs;
  int funccount;
  int n, lists;
  char **ptrs[MAX_STACK_ARGS] = {NULL};
  int nptrs[MAX_STACK_ARGS], i, maxi;

  if (nargs > 3) { /* Last arg must be the delimiter */
    n = nargs;
    lists = nargs - 2;
  } else {
    n = 4;
    lists = 2;
  }

  if (!delim_check(buff, bp, nargs, args, n, &sep))
    return;

  /* find our object and attribute */
  if (!fetch_ufun_attrib(args[0], executor, &ufun, UFUN_DEFAULT))
    return;

  maxi = 0;
  for (n = 0; n < lists; n++) {
    lp[n] = trim_space_sep(args[n + 1], sep);
    if (*lp[n]) {
      ptrs[n] = mush_calloc(MAX_SORTSIZE, sizeof(char *), "ptrarray");
      nptrs[n] = list2arr_ansi(ptrs[n], MAX_SORTSIZE, lp[n], sep, 1);
    } else {
      ptrs[n] = NULL;
      nptrs[n] = 0;
    }
    if (nptrs[n] > maxi) {
      maxi = nptrs[n];
    }
  }

  pe_regs = pe_regs_create(PE_REGS_ARG, "fun_mix");
  for (i = 0; i < maxi; i++) {
    for (n = 0; n < lists; n++) {
      if (nptrs[n] > i) {
        pe_regs_setenv_nocopy(pe_regs, n, ptrs[n][i]);
      } else {
        pe_regs_setenv_nocopy(pe_regs, n, "");
      }
    }
    funccount = pe_info->fun_invocations;
    if (call_ufun(&ufun, rbuff, executor, enactor, pe_info, pe_regs)) {
      break;
    }
    if (i > 0) {
      safe_chr(sep, buff, bp);
    }
    safe_str(rbuff, buff, bp);
    if (*bp == (buff + BUFFER_LEN - 1) &&
        pe_info->fun_invocations == funccount) {
      break;
    }
  }

  pe_regs_free(pe_regs);
  for (n = 0; n < lists; n++) {
    if (ptrs[n]) {
      freearr(ptrs[n], nptrs[n]);
      mush_free(ptrs[n], "ptrarray");
    }
  }
}

/* ARGSUSED */
FUNCTION(fun_table)
{
  /* TABLE(list, field_width, line_length, delimiter, output sep)
   * Given a list, produce a table (a column'd list)
   * Optional parameters: field width, line length, delimiter, output sep
   * Number of columns = line length / (field width+1)
   */
  size_t line_length = 78;
  size_t field_width = 10;
  size_t col = 0;
  size_t offset, col_len;
  char sep, osep, *cp, *t;
  char aligntype = '<';
  char *fwidth;
  ansi_string *as;

  if (!delim_check(buff, bp, nargs, args, 5, &osep))
    return;
  if ((nargs == 5) && !*args[4])
    osep = '\0';

  if (!delim_check(buff, bp, nargs, args, 4, &sep))
    return;

  if (nargs > 2) {
    if (!is_integer(args[2])) {
      safe_str(T(e_ints), buff, bp);
      return;
    }
    line_length = parse_integer(args[2]);
    if (line_length < 2)
      line_length = 2;
  }
  if (nargs > 1) {
    fwidth = args[1];
    if ((*fwidth) == '<' || (*fwidth) == '>' || (*fwidth) == '-') {
      aligntype = *(fwidth++);
    }
    if (!is_integer(fwidth)) {
      safe_str(T(e_ints), buff, bp);
      return;
    }
    field_width = parse_integer(fwidth);
    if (field_width < 1)
      field_width = 1;
    if (field_width >= BUFFER_LEN)
      field_width = BUFFER_LEN - 1;
  }
  if (field_width >= line_length)
    field_width = line_length - 1;

  /* Split out each token, truncate/pad it to field_width, and pack
   * it onto the line. When the line would go over line_length,
   * send a return
   */

  as = parse_ansi_string(args[0]);

  cp = trim_space_sep(as->text, sep);
  if (!*cp) {
    free_ansi_string(as);
    return;
  }

  t = split_token(&cp, sep);
  offset = t - &as->text[0];
  col_len = strlen(t);
  if (col_len > field_width)
    col_len = field_width;
  switch (aligntype) {
  case '<':
    safe_ansi_string(as, offset, col_len, buff, bp);
    if (safe_fill(' ', field_width - col_len, buff, bp)) {
      free_ansi_string(as);
      return;
    }
    break;
  case '>':
    safe_fill(' ', field_width - col_len, buff, bp);
    if (safe_ansi_string(as, offset, col_len, buff, bp)) {
      free_ansi_string(as);
      return;
    }
    break;
  case '-':
    safe_fill(' ', (field_width - col_len) / 2, buff, bp);
    safe_ansi_string(as, offset, col_len, buff, bp);
    if (safe_fill(' ', (field_width - col_len + 1) / 2, buff, bp)) {
      free_ansi_string(as);
      return;
    }
    break;
  }
  col = field_width;
  while (cp) {
    col += field_width;
    if (col > line_length) {
      safe_chr('\n', buff, bp);
      col = field_width;
    } else {
      if (osep) {
        safe_chr(osep, buff, bp);
        col += 1;
      }
    }
    t = split_token(&cp, sep);
    if (!t)
      break;
    offset = t - &as->text[0];
    col_len = strlen(t);
    if (col_len > field_width)
      col_len = field_width;
    switch (aligntype) {
    case '<':
      safe_ansi_string(as, offset, col_len, buff, bp);
      if (safe_fill(' ', field_width - col_len, buff, bp)) {
        free_ansi_string(as);
        return;
      }
      break;
    case '>':
      safe_fill(' ', field_width - col_len, buff, bp);
      if (safe_ansi_string(as, offset, col_len, buff, bp)) {
        free_ansi_string(as);
        return;
      }
      break;
    case '-':
      safe_fill(' ', (field_width - col_len) / 2, buff, bp);
      safe_ansi_string(as, offset, col_len, buff, bp);
      if (safe_fill(' ', (field_width - col_len + 1) / 2, buff, bp)) {
        free_ansi_string(as);
        return;
      }
      break;
    }
  }
  free_ansi_string(as);
}

/* string, regexp, replacement string. Acts like sed or perl's s///g,
 * with an ig version */
FUNCTION(fun_regreplace)
{
  pcre2_code *re;
  pcre2_match_data *md;
  int errcode;
  int subpatterns;
  PCRE2_SIZE erroffset;
  int flags = re_compile_flags, all = 0;
  PCRE2_SIZE match_offset = 0;
  PE_REGS *pe_regs = NULL;
  int i;
  const char *r, *obp;
  char *start;
  char tbuf[BUFFER_LEN], *tbp;
  char prebuf[BUFFER_LEN];
  char postbuf[BUFFER_LEN], *postp;
  ansi_string *orig = NULL;
  ansi_string *repl = NULL;
  PCRE2_SIZE search, prelen;
  size_t searchlen;
  int funccount;
  PCRE2_SIZE *offsets;

  if (called_as[strlen(called_as) - 1] == 'I') {
    flags |= PCRE2_CASELESS;
  }

  if (string_prefix(called_as, "REGEDITALL")) {
    all = 1;
  }

  /* Build orig */
  postp = postbuf;
  r = args[0];
  if (process_expression(postbuf, &postp, &r, executor, caller, enactor, eflags,
                         PT_DEFAULT, pe_info)) {
    return;
  }
  *postp = '\0';

  pe_regs = pe_regs_localize(pe_info, PE_REGS_REGEXP, "fun_regreplace");

  /* Ansi-less regedits */
  for (i = 1; i < nargs - 1 && !cpu_time_limit_hit; i += 2) {
    /* If this string has ANSI, switch to using ansi only */
    if (has_markup(postbuf)) {
      break;
    }

    memcpy(prebuf, postbuf, BUFFER_LEN);
    prelen = strlen(prebuf);

    postp = postbuf;

    /* Get the needle */
    tbp = tbuf;
    r = args[i];
    if (process_expression(tbuf, &tbp, &r, executor, caller, enactor, eflags,
                           PT_DEFAULT, pe_info)) {
      goto exit_sequence;
    }
    *tbp = '\0';

    if ((re =
           pcre2_compile((const PCRE2_UCHAR *) remove_markup(tbuf, &searchlen),
                         PCRE2_ZERO_TERMINATED, flags, &errcode, &erroffset,
                         re_compile_ctx)) == NULL) {
      /* Matching error. */
      char errstr[120];
      pcre2_get_error_message(errcode, (PCRE2_UCHAR *) errstr, sizeof errstr);
      safe_str(T("#-1 REGEXP ERROR: "), buff, bp);
      safe_str(errstr, buff, bp);
      goto exit_sequence;
    }
    ADD_CHECK("pcre"); /* re */
    if (searchlen) {
      searchlen--;
    }

    /* If we're doing a lot, jit the regexp to make it faster */
    if (all) {
      pcre2_jit_compile(re, PCRE2_JIT_COMPLETE);
    }

    md = pcre2_match_data_create_from_pattern(re, NULL);

    /* Do all the searches and replaces we can */

    start = prebuf;
    subpatterns = pcre2_match(re, (const PCRE2_UCHAR *) prebuf, prelen, 0,
                              re_match_flags, md, re_match_ctx);

    /* Match wasn't found... we're done */
    if (subpatterns < 0) {
      safe_str(prebuf, postbuf, &postp);
      pcre2_code_free(re);
      pcre2_match_data_free(md);
      DEL_CHECK("pcre");
      continue;
    }

    funccount = pe_info->fun_invocations;

    do {
      /* Copy up to the start of the matched area */
      char tmp;
      offsets = pcre2_get_ovector_pointer(md);
      tmp = prebuf[offsets[0]];
      prebuf[offsets[0]] = '\0';
      safe_str(start, postbuf, &postp);
      prebuf[offsets[0]] = tmp;

      /* Now copy in the replacement, putting in captured sub-expressions */
      obp = args[i + 1];

      pe_regs_clear(pe_regs);
      pe_regs_set_rx_context(pe_regs, 0, re, md, subpatterns);

      if (process_expression(postbuf, &postp, &obp, executor, caller, enactor,
                             eflags | PE_DOLLAR, PT_DEFAULT, pe_info)) {
        pcre2_code_free(re);
        pcre2_match_data_free(md);
        DEL_CHECK("pcre");
        goto exit_sequence;
      }
      if ((*bp == (buff + BUFFER_LEN - 1)) &&
          (pe_info->fun_invocations == funccount)) {
        break;
      }

      funccount = pe_info->fun_invocations;

      start = prebuf + offsets[1];
      match_offset = offsets[1];
      /* Make sure we advance at least 1 char */
      if (offsets[0] == match_offset) {
        match_offset++;
      }
    } while (all && match_offset < prelen && !cpu_time_limit_hit &&
             (subpatterns = pcre2_match(re, (const PCRE2_UCHAR *) prebuf,
                                        prelen, match_offset, re_match_flags,
                                        md, re_match_ctx)) >= 0);

    safe_str(start, postbuf, &postp);
    *postp = '\0';

    pcre2_code_free(re);
    pcre2_match_data_free(md);
    DEL_CHECK("pcre");
  }

  /* We get to this point if there is ansi in an 'orig' string */
  if (i < nargs - 1) {

    orig = parse_ansi_string(postbuf);

    /* For each search/replace pair, compare them against orig */
    for (; i < nargs - 1 && !cpu_time_limit_hit; i += 2) {

      /* Get the needle */
      tbp = tbuf;
      r = args[i];
      if (process_expression(tbuf, &tbp, &r, executor, caller, enactor, eflags,
                             PT_DEFAULT, pe_info)) {
        goto exit_sequence;
      }

      *tbp = '\0';

      if ((re = pcre2_compile(
             (const PCRE2_UCHAR *) remove_markup(tbuf, &searchlen),
             PCRE2_ZERO_TERMINATED, flags, &errcode, &erroffset,
             re_compile_ctx)) == NULL) {
        /* Matching error. */
        char errstr[120];
        pcre2_get_error_message(errcode, (PCRE2_UCHAR *) errstr, sizeof errstr);
        safe_str(T("#-1 REGEXP ERROR: "), buff, bp);
        safe_str(errstr, buff, bp);
        goto exit_sequence;
      }
      ADD_CHECK("pcre"); /* re */
      if (searchlen) {
        searchlen--;
      }

      if (all) {
        pcre2_jit_compile(re, PCRE2_JIT_COMPLETE);
      }
      md = pcre2_match_data_create_from_pattern(re, NULL);

      search = 0;
      /* Do all the searches and replaces we can */
      do {
        subpatterns =
          pcre2_match(re, (const PCRE2_UCHAR *) orig->text, orig->len, search,
                      re_match_flags, md, re_match_ctx);
        if (subpatterns >= 0) {
          /* We have a match */
          /* Process the replacement */
          r = args[i + 1];
          pe_regs_clear(pe_regs);
          pe_regs_set_rx_context_ansi(pe_regs, 0, re, md, subpatterns, orig);
          tbp = tbuf;
          if (process_expression(tbuf, &tbp, &r, executor, caller, enactor,
                                 eflags | PE_DOLLAR, PT_DEFAULT, pe_info)) {
            pcre2_code_free(re);
            pcre2_match_data_free(md);
            DEL_CHECK("pcre");
            goto exit_sequence;
          }
          *tbp = '\0';
          offsets = pcre2_get_ovector_pointer(md);
          if (offsets[0] >= search) {
            repl = parse_ansi_string(tbuf);

            /* Do the replacement */
            ansi_string_replace(orig, offsets[0], offsets[1] - offsets[0],
                                repl);

            /* Advance search */
            if (search == offsets[1]) {
              search = offsets[0] + repl->len;
              search++;
            } else {
              search = offsets[0] + repl->len;
            }
            /* if (offsets[0] < 1) search++; */

            free_ansi_string(repl);
            if (search >= (PCRE2_SIZE) orig->len) {
              break;
            }
          } else {
            break;
          }
        }
      } while (subpatterns >= 0 && !cpu_time_limit_hit && all);
      pcre2_code_free(re);
      pcre2_match_data_free(md);
      DEL_CHECK("pcre");
    }
    safe_ansi_string(orig, 0, orig->len, buff, bp);
    free_ansi_string(orig);
    orig = NULL;
  } else {
    safe_str(postbuf, buff, bp);
  }

exit_sequence:
  if (orig) {
    free_ansi_string(orig);
  }
  pe_regs_restore(pe_info, pe_regs);
  pe_regs_free(pe_regs);
}

FUNCTION(fun_regmatch)
{
  /* ---------------------------------------------------------------------------
   * fun_regmatch Return 0 or 1 depending on whether or not a regular
   * expression matches a string. If a third argument is specified, dump
   * the results of a regexp pattern match into a set of r()-registers.
   *
   * regmatch(string, pattern, list of registers)
   * Registers are by position (old way) or name:register (new way)
   *
   */
  int i, nqregs;
  char *qregs[NUMQ], *holder[NUMQ];
  pcre2_code *re;
  pcre2_match_data *md;
  int errcode;
  PCRE2_SIZE erroffset;
  const char *errptr = NULL;
  int subpatterns;
  char lbuff[BUFFER_LEN], *lbp;
  int flags = re_compile_flags;
  ansi_string *as;
  PCRE2_UCHAR *needle, *txt;
  size_t len;

  if (strcmp(called_as, "REGMATCHI") == 0) {
    flags |= PCRE2_CASELESS;
  }

  needle = (PCRE2_UCHAR *) remove_markup(args[1], &len);

  as = parse_ansi_string(args[0]);
  txt = (PCRE2_UCHAR *) as->text;
  if (nargs == 2) { /* Don't care about saving sub expressions */
    bool match = quick_regexp_match((const char *) needle, (const char *) txt,
                                    (flags & PCRE2_CASELESS) ? 0 : 1, &errptr);
    if (errptr == NULL) {
      safe_boolean(match, buff, bp);
    } else {
      safe_str(T("#-1 REGEXP ERROR: "), buff, bp);
      safe_str(errptr, buff, bp);
    }
    free_ansi_string(as);
    return;
  }

  if ((re = pcre2_compile(needle, PCRE2_ZERO_TERMINATED, flags, &errcode,
                          &erroffset, re_compile_ctx)) == NULL) {
    char errstr[120];
    /* Matching error. */
    pcre2_get_error_message(errcode, (PCRE2_UCHAR *) errstr, sizeof errstr);
    safe_str(T("#-1 REGEXP ERROR: "), buff, bp);
    safe_str(errstr, buff, bp);
    free_ansi_string(as);
    return;
  }
  ADD_CHECK("pcre");
  md = pcre2_match_data_create_from_pattern(re, NULL);

  subpatterns =
    pcre2_match(re, txt, as->len, 0, re_match_flags, md, re_match_ctx);
  safe_integer(subpatterns >= 0, buff, bp);

  /* We need to parse the list of registers.  Anything that we don't parse
   * is assumed to be -1.  If we didn't match, or the match went wonky,
   * then set the register to empty.  Otherwise, fill the register with
   * the subexpression.
   */
  nqregs = list2arr(qregs, NUMQ, args[2], ' ', 1);

  /* Initialize every q-register used to '' */
  for (i = 0; i < nqregs; i++) {
    char *regname;
    holder[i] = mush_strdup(qregs[i], "regmatch");
    if ((regname = strchr(holder[i], ':'))) {
      /* subexpr:register */
      *regname++ = '\0';
    } else {
      /* Get subexper by position in list */
      regname = holder[i];
    }

    if (ValidQregName(regname)) {
      PE_Setq(pe_info, regname, "");
    }
  }
  /* Now, only for those that have a pattern, copy text */
  for (i = 0; i < nqregs; i++) {
    char *regname;
    char *named_subpattern = NULL;
    int subpattern = 0;
    if ((regname = strchr(qregs[i], ':'))) {
      /* subexpr:register */
      *regname++ = '\0';
      if (is_strict_integer(qregs[i]))
        subpattern = parse_integer(qregs[i]);
      else
        named_subpattern = qregs[i];
    } else {
      /* Get subexper by position in list */
      subpattern = i;
      regname = qregs[i];
    }

    if (!ValidQregName(regname)) {
      if (regname[0] != '-' || regname[1]) {
        safe_str(T(e_badregname), buff, bp);
      }
      continue;
    }

    if (subpatterns < 0) {
      lbuff[0] = '\0';
    } else if (named_subpattern) {
      lbp = lbuff;
      ansi_pcre_copy_named_substring(re, as, md, subpatterns, named_subpattern,
                                     1, lbuff, &lbp);
      *(lbp) = '\0';
    } else {
      lbp = lbuff;
      ansi_pcre_copy_substring(as, md, subpatterns, subpattern, 1, lbuff, &lbp);
      *(lbp) = '\0';
    }
    PE_Setq(pe_info, regname, lbuff);
  }
  for (i = 0; i < nqregs; i++) {
    mush_free(holder[i], "regmatch");
  }
  pcre2_code_free(re);
  pcre2_match_data_free(md);
  DEL_CHECK("pcre");
  free_ansi_string(as);
}

/* Like grab, but with a regexp pattern. This same function handles
 * regrab(), regraball(), and the case-insenstive versions,
 * as well as reglmatch() and reglmatchall(). */
FUNCTION(fun_regrab)
{
  char *r, *s, *b, sep;
  size_t rlen;
  pcre2_code *re;
  pcre2_match_data *md;
  int errcode;
  PCRE2_SIZE erroffset;
  int flags = re_compile_flags;
  char *osep, osepd[2] = {'\0', '\0'};
  char **ptrs;
  int nptrs, i;
  bool all = 0, pos = 0;

  if (!delim_check(buff, bp, nargs, args, 3, &sep)) {
    return;
  }

  if (nargs == 4) {
    osep = args[3];
  } else {
    osepd[0] = sep;
    osep = osepd;
  }

  s = trim_space_sep(args[0], sep);
  b = *bp;

  if (strrchr(called_as, 'I')) {
    flags |= PCRE2_CASELESS;
  }

  if (strstr(called_as, "ALL")) {
    all = 1;
  }

  if (strstr(called_as, "MATCH")) {
    pos = 1;
  }

  if ((re = pcre2_compile((const PCRE2_UCHAR *) remove_markup(args[1], NULL),
                          PCRE2_ZERO_TERMINATED, flags, &errcode, &erroffset,
                          re_compile_ctx)) == NULL) {
    /* Matching error. */
    char errstr[120];
    pcre2_get_error_message(errcode, (PCRE2_UCHAR *) errstr, sizeof errstr);
    safe_str(T("#-1 REGEXP ERROR: "), buff, bp);
    safe_str(errstr, buff, bp);
    return;
  }
  ADD_CHECK("pcre");
  md = pcre2_match_data_create_from_pattern(re, NULL);

  ptrs = mush_calloc(MAX_SORTSIZE, sizeof(char *), "ptrarray");
  if (!ptrs) {
    mush_panic("Unable to allocate memory in fun_regrab");
  }
  nptrs = list2arr_ansi(ptrs, MAX_SORTSIZE, s, sep, 1);
  for (i = 0; i < nptrs && !cpu_time_limit_hit; i++) {
    r = remove_markup(ptrs[i], &rlen);
    if (pcre2_match(re, (const PCRE2_UCHAR *) r, rlen - 1, 0, re_match_flags,
                    md, re_match_ctx) >= 0) {
      if (all && *bp != b) {
        safe_str(osep, buff, bp);
      }
      if (pos) {
        safe_integer(i + 1, buff, bp);
      } else {
        safe_str(ptrs[i], buff, bp);
      }
      if (!all) {
        break;
      }
    }
  }
  freearr(ptrs, nptrs);
  mush_free(ptrs, "ptrarray");

  pcre2_code_free(re);
  pcre2_match_data_free(md);
  DEL_CHECK("pcre");
}

FUNCTION(fun_isregexp)
{
  pcre2_code *re;
  int errcode;
  PCRE2_SIZE erroffset;

  if (!!(re = pcre2_compile((const PCRE2_UCHAR *) args[0], arglens[0],
                            re_compile_flags, &errcode, &erroffset,
                            re_compile_ctx))) {
    pcre2_code_free(re);
    safe_chr('1', buff, bp);
    return;
  }
  safe_chr('0', buff, bp);
}
