/**
 * \file funmath.c
 *
 * \brief Mathematical functions for mushcode.
 *
 *
 */

#include "copyrite.h"

#include <math.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#ifdef HAVE_FENV_H
#include <fenv.h>
#endif

#ifdef HAVE_SSE2
#include <emmintrin.h>
#endif
#ifdef HAVE_SSE3
#include <pmmintrin.h>
#endif

#include "conf.h"
#include "externs.h"
#include "mymalloc.h"
#include "notify.h"
#include "parse.h"
#include "sort.h"
#include "strutil.h"

#ifdef WIN32
#pragma warning(disable : 4761) /* NJG: disable warning re conversion */
#endif

#ifndef M_PI
/** The ratio of the circumference of a circle to its denominator. */
#define M_PI 3.14159265358979323846264338327
#endif

#ifndef M_PI_2
#define M_PI_2 1.57079632679489661923 /* pi/2 */
#endif

#define EPSILON 0.000000001 /**< limit of precision for float equality */
#define EQE(x, y, e) (fabs(x - y) < e) /**< floating point equality macro */
#define EQ(x, y) EQE(x, y, EPSILON)    /**< floating point equality macro */

static void do_spellnum(char *num, unsigned int len, char **buff, char ***bp);
static void do_ordinalize(char **buff, char ***bp);
static NVAL find_median(NVAL *, int);

static NVAL angle_to_rad(NVAL angle, const char *from);
static NVAL rad_to_angle(NVAL angle, const char *to);
static double frac(double v, double *RESTRICT n, double *RESTRICT d,
                   double error);

int format_long(intmax_t n, char *buff, char **bp, int maxlen, int base);
static void lmathcomp(char **ptr, int nptr, char *buff, char **bp, int eqokay,
                      int isgt);

/* Generated by gperf */
#include "lmathtab.c"

/* Functions for testing and parsing IVALs and UIVALs, the types of
 * arguments to math functions that work on integers instead of
 *  floating-point numbers. No matter what IVAL is (32-bit or 64-bit),
 *  they can be passed to safe_integer()/safe_uinteger().
 *
 * Math functions that operate on IVALs: div(), floordiv(), modulo(),
 *  remainder()
 * Math functions that operate on UIVALS: shl(), shr(), band(), bnot(), bor()
 *  bxor(), bnand()
 *
 * Other functions work on NVALs or accept plain ints
 */

static IVAL
parse_ival_full(const char *str, char **end, int base)
{
  return parse_int64(str, end, base);
}

static IVAL
parse_ival(const char *str)
{
  return parse_ival_full(str, NULL, 10);
}

static UIVAL
parse_uival_full(const char *str, char **end, int base)
{
  return parse_uint64(str, end, base);
}

static UIVAL
parse_uival(const char *str)
{
  return parse_uival_full(str, NULL, 10);
}

/** Is string an integer suitable for a math function?
 * To TinyMUSH, any string is an integer. To PennMUSH, a string that
 * passes strtol is an integer, and a blank string is an integer
 * if NULL_EQ_ZERO is turned on.
 * \param str string to check.
 * \retval 1 string is an integer.
 * \retval 0 string is not an integer.
 */
static bool
is_ival(char const *str)
{
  char *end;

  /* If we're emulating Tiny, anything is an integer */
  if (TINY_MATH)
    return 1;
  if (!str)
    return 0;
  while (isspace(*str))
    str++;
  if (*str == '\0')
    return NULL_EQ_ZERO;
  errno = 0;
  parse_ival_full(str, &end, 10);
  if (errno == ERANGE || *end != '\0')
    return 0;
  return 1;
}

/** Is string a UIVAL?
 * To TinyMUSH, any string is an uinteger. To PennMUSH, a string that
 * passes strtoul is an uinteger, and a blank string is an uinteger
 * if NULL_EQ_ZERO is turned on.
 * \param str string to check.
 * \retval 1 string is an uinteger.
 * \retval 0 string is not an uinteger.
 */
static bool
is_uival(char const *str)
{
  char *end;

  /* If we're emulating Tiny, anything is an integer */
  if (TINY_MATH)
    return 1;
  if (!str)
    return 0;
  /* strtoul() accepts negative numbers, so we still have to do this check */
  while (isspace(*str))
    str++;
  if (*str == '\0')
    return NULL_EQ_ZERO;
  if (!(isdigit(*str) || *str == '+'))
    return 0;
  errno = 0;
  parse_uival_full(str, &end, 10);
  if (errno == ERANGE || *end != '\0')
    return 0;
  return 1;
}

/* ARGSUSED */
FUNCTION(fun_ctu)
{
  NVAL angle;
  if (!is_number(args[0])) {
    safe_str(T(e_num), buff, bp);
    return;
  }

  if (!args[1] || !args[2]) {
    safe_str(T("#-1 INVALID ANGLE TYPE"), buff, bp);
    return;
  }
  angle = angle_to_rad(parse_number(args[0]), args[1]);
  safe_number(rad_to_angle(angle, args[2]), buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_add) { math_add(args, nargs, buff, bp); }

/* ARGSUSED */
FUNCTION(fun_sub) { math_sub(args, nargs, buff, bp); }

/* ARGSUSED */
FUNCTION(fun_mul) { math_mul(args, nargs, buff, bp); }

/* TO-DO: I have better code for comparing floating-point numbers
   lying around somewhere. The idea is that numbers that are very
   close can be 'close enough' to be equal or whatever, without having
   to be exactly the same. */

/* ARGSUSED */
FUNCTION(fun_gt) { math_gt(args, nargs, buff, bp); }

/* ARGSUSED */
FUNCTION(fun_gte) { math_gte(args, nargs, buff, bp); }

/* ARGSUSED */
FUNCTION(fun_lt) { math_lt(args, nargs, buff, bp); }

/* ARGSUSED */
FUNCTION(fun_lte) { math_lte(args, nargs, buff, bp); }

/* ARGSUSED */
FUNCTION(fun_eq) { math_eq(args, nargs, buff, bp); }

/* ARGSUSED */
FUNCTION(fun_neq) { math_neq(args, nargs, buff, bp); }

/* ARGSUSED */
FUNCTION(fun_max) { math_max(args, nargs, buff, bp); }

/* ARGSUSED */
FUNCTION(fun_min) { math_min(args, nargs, buff, bp); }

/* ARGSUSED */
FUNCTION(fun_sign)
{
  NVAL x;

  if (!is_number(args[0])) {
    safe_str(T(e_num), buff, bp);
    return;
  }
  x = parse_number(args[0]);
  if (EQ(x, 0))
    safe_chr('0', buff, bp);
  else if (x > 0)
    safe_chr('1', buff, bp);
  else
    safe_str("-1", buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_shl)
{
  if (!is_uival(args[0]) || !is_uival(args[1])) {
    safe_str(T(e_uints), buff, bp);
    return;
  }
  safe_uinteger(parse_uival(args[0]) << parse_uival(args[1]), buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_shr)
{
  if (!is_uival(args[0]) || !is_uival(args[1])) {
    safe_str(T(e_uints), buff, bp);
    return;
  }
  safe_uinteger(parse_uival(args[0]) >> parse_uival(args[1]), buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_inc)
{
  int num;
  char *p;
  /* Handle the case of a pure number */
  if (is_strict_integer(args[0])) {
    safe_integer(parse_integer(args[0]) + 1, buff, bp);
    return;
  }
  /* Handle a null string */
  if (!*args[0]) {
    safe_str(NULL_EQ_ZERO ? "1" : T("#-1 ARGUMENT MUST END IN AN INTEGER"),
             buff, bp);
    return;
  }
  p = args[0] + arglens[0] - 1;
  if (!isdigit(*p)) {
    if (NULL_EQ_ZERO) {
      safe_str(args[0], buff, bp);
      safe_str("1", buff, bp);
    } else
      safe_str(T("#-1 ARGUMENT MUST END IN AN INTEGER"), buff, bp);
    return;
  }
  while ((isdigit(*p) || (*p == '-')) && p != args[0]) {
    if (*p == '-') {
      p--;
      break;
    }
    p--;
  }
  /* p now points to the last non-numeric character in the string */
  if (p == args[0] && (isdigit(*p) || (*p == '-'))) {
    /* Special case - it's all digits, but out of range. */
    safe_str(T(e_range), buff, bp);
    return;
  }

  /* Move it to the first numeric character */
  p++;
  num = parse_integer(p) + 1;
  *p = '\0';
  safe_str(args[0], buff, bp);
  safe_integer(num, buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_dec)
{
  int num;
  char *p;
  /* Handle the case of a pure number */
  if (is_strict_integer(args[0])) {
    safe_integer(parse_integer(args[0]) - 1, buff, bp);
    return;
  }
  /* Handle a null string */
  if (!*args[0]) {
    safe_str(NULL_EQ_ZERO ? "-1" : T("#-1 ARGUMENT MUST END IN AN INTEGER"),
             buff, bp);
    return;
  }
  p = args[0] + arglens[0] - 1;
  if (!isdigit(*p)) {
    if (NULL_EQ_ZERO) {
      safe_str(args[0], buff, bp);
      safe_str("-1", buff, bp);
    } else
      safe_str(T("#-1 ARGUMENT MUST END IN AN INTEGER"), buff, bp);
    return;
  }
  while ((isdigit(*p) || (*p == '-')) && p != args[0]) {
    if (*p == '-') {
      p--;
      break;
    }
    p--;
  }
  /* p now points to the last non-numeric character in the string */
  if (p == args[0] && (isdigit(*p) || (*p == '-'))) {
    /* Special case - it's all digits, but out of range. */
    safe_str(T(e_range), buff, bp);
    return;
  }
  /* Move it to the first numeric character */
  p++;
  num = parse_integer(p) - 1;
  *p = '\0';
  safe_str(args[0], buff, bp);
  safe_integer(num, buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_trunc)
{
  /* This function does not have the non-number check because
   * the help file explicitly states that this function can
   * be used to turn "101dalmations" into "101".
   */
  safe_integer(parse_integer(args[0]), buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_div) { math_div(args, nargs, buff, bp); }

/* ARGSUSED */
FUNCTION(fun_floordiv) { math_floordiv(args, nargs, buff, bp); }

/* ARGSUSED */
FUNCTION(fun_modulo) { math_modulo(args, nargs, buff, bp); }

/* ARGSUSED */
FUNCTION(fun_remainder) { math_remainder(args, nargs, buff, bp); }

/* ARGSUSED */
FUNCTION(fun_abs)
{
  if (!is_number(args[0])) {
    safe_str(T(e_num), buff, bp);
    return;
  }
  safe_number(fabs(parse_number(args[0])), buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_dist2d) { math_dist2d(args, nargs, buff, bp); }

FUNCTION(fun_dist3d) { math_dist3d(args, nargs, buff, bp); }

/* ------------------------------------------------------------------------
 * Dune's vector functions: VADD, VSUB, VMUL, VCROSS, VMAG, VUNIT, VDIM
 *  VCRAMER?
 * Vectors are space-separated numbers.
 */

/* ARGSUSED */
FUNCTION(fun_vmax)
{
  char *p1, *p2;
  char *start;
  char sep;
  NVAL a, b;

  /* return if a list is empty */
  if (!args[0] || !args[1]) {
    safe_str(T("#-1 VECTORS MUST BE SAME DIMENSIONS"), buff, bp);
    return;
  }

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;
  p1 = trim_space_sep(args[0], sep);
  p2 = trim_space_sep(args[1], sep);

  /* return if a list is empty */
  if (!*p1 || !*p2) {
    safe_str(T("#-1 VECTORS MUST BE SAME DIMENSIONS"), buff, bp);
    return;
  }

  /* max the vectors */
  start = *bp;
  a = parse_number(split_token(&p1, sep));
  b = parse_number(split_token(&p2, sep));
  safe_number((a > b) ? a : b, buff, bp);

  while (p1 && p2) {
    safe_chr(sep, buff, bp);
    a = parse_number(split_token(&p1, sep));
    b = parse_number(split_token(&p2, sep));
    safe_number((a > b) ? a : b, buff, bp);
  }

  /* make sure vectors were the same length */
  if (p1 || p2) {
    *bp = start;
    safe_str(T("#-1 VECTORS MUST BE SAME DIMENSIONS"), buff, bp);
    return;
  }
}

/* ARGSUSED */
FUNCTION(fun_vmin)
{
  char *p1, *p2;
  char *start;
  char sep;
  NVAL a, b;

  /* return if a list is empty */
  if (!args[0] || !args[1]) {
    safe_str(T("#-1 VECTORS MUST BE SAME DIMENSIONS"), buff, bp);
    return;
  }

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;
  p1 = trim_space_sep(args[0], sep);
  p2 = trim_space_sep(args[1], sep);

  /* return if a list is empty */
  if (!*p1 || !*p2)
    return;

  /* max the vectors */
  start = *bp;
  a = parse_number(split_token(&p1, sep));
  b = parse_number(split_token(&p2, sep));
  safe_number((a < b) ? a : b, buff, bp);

  while (p1 && p2) {
    safe_chr(sep, buff, bp);
    a = parse_number(split_token(&p1, sep));
    b = parse_number(split_token(&p2, sep));
    safe_number((a < b) ? a : b, buff, bp);
  }

  /* make sure vectors were the same length */
  if (p1 || p2) {
    *bp = start;
    safe_str(T("#-1 VECTORS MUST BE SAME DIMENSIONS"), buff, bp);
    return;
  }
}

/* ARGSUSED */
FUNCTION(fun_vadd)
{
  char *p1, *p2;
  char *start;
  char sep;

  /* return if a list is empty */
  if (!args[0] || !args[1]) {
    safe_str(T("#-1 VECTORS MUST BE SAME DIMENSIONS"), buff, bp);
    return;
  }

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;
  p1 = trim_space_sep(args[0], sep);
  p2 = trim_space_sep(args[1], sep);

  /* return if a list is empty */
  if (!*p1 || !*p2) {
    safe_str(T("#-1 VECTORS MUST BE SAME DIMENSIONS"), buff, bp);
    return;
  }

  /* add the vectors */
  start = *bp;
  safe_number(parse_number(split_token(&p1, sep)) +
                parse_number(split_token(&p2, sep)),
              buff, bp);
  while (p1 && p2) {
    safe_chr(sep, buff, bp);
    safe_number(parse_number(split_token(&p1, sep)) +
                  parse_number(split_token(&p2, sep)),
                buff, bp);
  }

  /* make sure vectors were the same length */
  if (p1 || p2) {
    *bp = start;
    safe_str(T("#-1 VECTORS MUST BE SAME DIMENSIONS"), buff, bp);
    return;
  }
}

/* ARGSUSED */
FUNCTION(fun_vsub)
{
  char *p1, *p2;
  char *start;
  char sep;

  /* return if a list is empty */
  if (!args[0] || !args[1]) {
    safe_str(T("#-1 VECTORS MUST BE SAME DIMENSIONS"), buff, bp);
    return;
  }

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;
  p1 = trim_space_sep(args[0], sep);
  p2 = trim_space_sep(args[1], sep);

  /* return if a list is empty */
  if (!*p1 || !*p2) {
    safe_str(T("#-1 VECTORS MUST BE SAME DIMENSIONS"), buff, bp);
    return;
  }

  /* subtract the vectors */
  start = *bp;
  safe_number(parse_number(split_token(&p1, sep)) -
                parse_number(split_token(&p2, sep)),
              buff, bp);
  while (p1 && p2) {
    safe_chr(sep, buff, bp);
    safe_number(parse_number(split_token(&p1, sep)) -
                  parse_number(split_token(&p2, sep)),
                buff, bp);
  }

  /* make sure vectors were the same length */
  if (p1 || p2) {
    *bp = start;
    safe_str(T("#-1 VECTORS MUST BE SAME DIMENSIONS"), buff, bp);
    return;
  }
}

/* ARGSUSED */
FUNCTION(fun_vmul)
{
  NVAL e1, e2;
  char *p1, *p2;
  char *start;
  char sep;

  /* return if a list is empty */
  if (!args[0] || !args[1]) {
    safe_str(T("#-1 VECTORS MUST BE SAME DIMENSIONS"), buff, bp);
    return;
  }

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;
  p1 = trim_space_sep(args[0], sep);
  p2 = trim_space_sep(args[1], sep);

  /* return if a list is empty */
  if (!*p1 || !*p2) {
    safe_str(T("#-1 VECTORS MUST BE SAME DIMENSIONS"), buff, bp);
    return;
  }

  /* multiply the vectors */
  start = *bp;
  e1 = parse_number(split_token(&p1, sep));
  e2 = parse_number(split_token(&p2, sep));
  if (!p1) {
    /* scalar * vector */
    safe_number(e1 * e2, buff, bp);
    while (p2) {
      safe_chr(sep, buff, bp);
      safe_number(e1 * parse_number(split_token(&p2, sep)), buff, bp);
    }
  } else if (!p2) {
    /* vector * scalar */
    safe_number(e1 * e2, buff, bp);
    while (p1) {
      safe_chr(sep, buff, bp);
      safe_number(parse_number(split_token(&p1, sep)) * e2, buff, bp);
    }
  } else {
    /* vector * vector elementwise product */
    safe_number(e1 * e2, buff, bp);
    while (p1 && p2) {
      safe_chr(sep, buff, bp);
      safe_number(parse_number(split_token(&p1, sep)) *
                    parse_number(split_token(&p2, sep)),
                  buff, bp);
    }
    /* make sure vectors were the same length */
    if (p1 || p2) {
      *bp = start;
      safe_str(T("#-1 VECTORS MUST BE SAME DIMENSIONS"), buff, bp);
      return;
    }
  }
}

/* ARGSUSED */
FUNCTION(fun_vdot)
{
  NVAL product;
  char *p1, *p2;
  char sep;

  /* return if a list is empty */
  if (!args[0] || !args[1]) {
    safe_str(T("#-1 VECTORS MUST BE SAME DIMENSIONS"), buff, bp);
    return;
  }

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;
  p1 = trim_space_sep(args[0], sep);
  p2 = trim_space_sep(args[1], sep);

  /* return if a list is empty */
  if (!*p1 || !*p2) {
    safe_str(T("#-1 VECTORS MUST BE SAME DIMENSIONS"), buff, bp);
    return;
  }

  /* multiply the vectors */
  product = 0;
  while (p1 && p2) {
    product +=
      parse_number(split_token(&p1, sep)) * parse_number(split_token(&p2, sep));
  }
  if (p1 || p2) {
    safe_str(T("#-1 VECTORS MUST BE SAME DIMENSIONS"), buff, bp);
    return;
  }
  safe_number(product, buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_vmag)
{
  NVAL num, sum;
  char *p1;
  char sep;

  /* return if a list is empty */
  if (!args[0]) {
    safe_str(T("#-1 VECTOR MUST NOT BE EMPTY"), buff, bp);
    return;
  }

  if (!delim_check(buff, bp, nargs, args, 2, &sep))
    return;
  p1 = trim_space_sep(args[0], sep);

  /* return if a list is empty */
  if (!*p1) {
    safe_str(T("#-1 VECTOR MUST NOT BE EMPTY"), buff, bp);
    return;
  }

  /* sum the squares */
  num = parse_number(split_token(&p1, sep));
  sum = num * num;
  while (p1) {
    num = parse_number(split_token(&p1, sep));
    sum += num * num;
  }

  safe_number(sqrt(sum), buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_vunit)
{
  NVAL num, sum;
  char tbuf[BUFFER_LEN];
  char *p1;
  char sep;

  /* return if a list is empty */
  if (!args[0]) {
    safe_str(T("#-1 VECTOR MUST NOT BE EMPTY"), buff, bp);
    return;
  }

  if (!delim_check(buff, bp, nargs, args, 2, &sep))
    return;
  p1 = trim_space_sep(args[0], sep);

  /* return if a list is empty */
  if (!*p1) {
    safe_str(T("#-1 VECTOR MUST NOT BE EMPTY"), buff, bp);
    return;
  }

  /* copy the vector, since we have to walk it twice... */
  strcpy(tbuf, p1);

  /* find the magnitude */
  num = parse_number(split_token(&p1, sep));
  sum = num * num;
  while (p1) {
    num = parse_number(split_token(&p1, sep));
    sum += num * num;
  }
  sum = sqrt(sum);

  if (EQ(sum, 0)) {
    /* zero vector */
    p1 = tbuf;
    safe_chr('0', buff, bp);
    while (split_token(&p1, sep), p1) {
      safe_chr(sep, buff, bp);
      safe_chr('0', buff, bp);
    }
    return;
  }
  /* now make the unit vector */
  p1 = tbuf;
  safe_number(parse_number(split_token(&p1, sep)) / sum, buff, bp);
  while (p1) {
    safe_chr(sep, buff, bp);
    safe_number(parse_number(split_token(&p1, sep)) / sum, buff, bp);
  }
}

FUNCTION(fun_vcross)
{
  char sep = ' ';
  char *v1[BUFFER_LEN / 2], *v2[BUFFER_LEN / 2];
  int v1len, v2len, n;
  NVAL vec1[3], vec2[3], cross[3];

  if (!delim_check(buff, bp, nargs, args, 3, &sep))
    return;

  v1len = list2arr(v1, BUFFER_LEN / 2, args[0], sep, 1);
  v2len = list2arr(v2, BUFFER_LEN / 2, args[1], sep, 1);

  if (v1len != 3 || v2len != 3) {
    safe_str(T("#-1 VECTORS MUST BE THREE-DIMENSIONAL"), buff, bp);
    return;
  }

  for (n = 0; n < 3; n++) {
    vec1[n] = parse_number(v1[n]);
    vec2[n] = parse_number(v2[n]);
  }

#ifdef HAVE_SSE2
  {
    __m128d xy1, xy2, xy3, xy4, z1, z2;

    xy1 = _mm_set_pd(vec1[2], vec1[1]);
    xy2 = _mm_set_pd(vec2[0], vec2[2]);
    xy3 = _mm_set_pd(vec2[2], vec2[1]);
    xy4 = _mm_set_pd(vec1[0], vec1[2]);
    z1 = _mm_set_pd(vec2[0], vec1[0]);
    z2 = _mm_set_pd(vec1[1], vec2[1]);

    xy1 = _mm_mul_pd(xy1, xy2);
    xy2 = _mm_mul_pd(xy3, xy4);
    xy3 = _mm_sub_pd(xy1, xy2);
    z1 = _mm_mul_pd(z1, z2);

#ifdef HAVE_SSE3
    z1 = _mm_hsub_pd(z1, z2);
    _mm_store_sd(cross + 2, z1);
#else
    {
      /* SSE2 version */
      double zsub[2] __attribute__((__aligned__(16)));
      _mm_store_pd(zsub, z1);
      cross[2] = zsub[0] - zsub[1];
    }
#endif

    _mm_store_pd(cross, xy3);
  }
#else
  /* Scalar version */
  cross[0] = vec1[1] * vec2[2] - vec2[1] * vec1[2];
  cross[1] = vec1[2] * vec2[0] - vec2[2] * vec1[0];
  cross[2] = vec1[0] * vec2[1] - vec2[0] * vec1[1];
#endif

  safe_number(cross[0], buff, bp);
  safe_chr(sep, buff, bp);
  safe_number(cross[1], buff, bp);
  safe_chr(sep, buff, bp);
  safe_number(cross[2], buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_fdiv) { math_fdiv(args, nargs, buff, bp); }

/* ARGSUSED */
FUNCTION(fun_fmod)
{
  NVAL x, y, m;
  if (!is_strict_number(args[0]) || !is_strict_number(args[1])) {
    safe_str(T(e_nums), buff, bp);
    return;
  }
  y = parse_number(args[1]);
  if (EQ(y, 0)) {
    safe_str(T("#-1 DIVISION BY ZERO"), buff, bp);
    return;
  }
  x = parse_number(args[0]);

#ifdef HAVE_FECLEAREXCEPT
  errno = 0;
  feclearexcept(FE_ALL_EXCEPT);
#endif

  m = fmod(x, y);

#ifdef HAVE_FETESTEXCEPT
  if (errno ||
      fetestexcept(FE_INVALID | FE_OVERFLOW | FE_UNDERFLOW | FE_DIVBYZERO)) {
    safe_str(T(e_range), buff, bp);
    return;
  }
#endif

  safe_number(m, buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_floor)
{
  if (!is_number(args[0])) {
    safe_str(T(e_num), buff, bp);
    return;
  }
  safe_number(floor(parse_number(args[0])), buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_ceil)
{
  if (!is_number(args[0])) {
    safe_str(T(e_num), buff, bp);
    return;
  }
  safe_number(ceil(parse_number(args[0])), buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_round)
{
  unsigned int places;
  double n;
  char *sbp, *decimal;
  bool pad = 0;

  if (!is_number(args[0])) {
    safe_str(T(e_num), buff, bp);
    return;
  } else
    n = parse_number(args[0]);

  if (nargs >= 2) {
    if (!is_uinteger(args[1])) {
      safe_str(T(e_int), buff, bp);
      return;
    }
    places = parse_uinteger(args[1]);
  } else
    places = 0;

  if (nargs == 3)
    pad = parse_boolean(args[2]);

  if (places > (unsigned int) FLOAT_PRECISION)
    places = FLOAT_PRECISION;

#ifdef HAVE_LRINT
  if (places == 0)
    safe_integer(lrint(n), buff, bp);
  else
#endif
  {
    sbp = *bp;
    safe_format(buff, bp, "%.*f", places, n);
    *(*bp) = '\0';

    decimal = strchr(sbp, '.');
    if (!pad && decimal && places > 1) {
      int i, trailing;
      decimal += 2;
      trailing = strlen(decimal);
      for (i = 0; i < trailing; i++, decimal++) {
        size_t len = strspn(decimal, "0");
        if (*(decimal + len) == '\0') {
          *bp = decimal;
          break;
        }
      }
    }
  }
}

/* ARGSUSED */
FUNCTION(fun_pi) { safe_number(M_PI, buff, bp); }

/* ARGSUSED */
FUNCTION(fun_sin)
{
  NVAL angle, s;
  if (!is_number(args[0])) {
    safe_str(T(e_num), buff, bp);
    return;
  }
  angle = angle_to_rad(parse_number(args[0]), args[1]);

#ifdef HAVE_FECLEAREXCEPT
  errno = 0;
  feclearexcept(FE_ALL_EXCEPT);
#endif

  s = sin(angle);

#ifdef HAVE_FETESTEXCEPT
  if (errno ||
      fetestexcept(FE_INVALID | FE_OVERFLOW | FE_UNDERFLOW | FE_DIVBYZERO)) {
    safe_str(T(e_range), buff, bp);
    return;
  }
#endif

  safe_number(s, buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_asin)
{
  NVAL num;
  if (!is_number(args[0])) {
    safe_str(T(e_num), buff, bp);
    return;
  }
  num = parse_number(args[0]);
  if ((num < -1) || (num > 1)) {
    safe_str(T(e_range), buff, bp);
    return;
  }
  safe_number(rad_to_angle(asin(num), args[1]), buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_cos)
{
  NVAL angle, c;
  if (!is_number(args[0])) {
    safe_str(T(e_num), buff, bp);
    return;
  }
  angle = angle_to_rad(parse_number(args[0]), args[1]);

#ifdef HAVE_FECLEAREXCEPT
  errno = 0;
  feclearexcept(FE_ALL_EXCEPT);
#endif

  c = cos(angle);

#ifdef HAVE_FETESTEXCEPT
  if (errno ||
      fetestexcept(FE_INVALID | FE_OVERFLOW | FE_UNDERFLOW | FE_DIVBYZERO)) {
    safe_str(T(e_range), buff, bp);
    return;
  }
#endif

  safe_number(c, buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_acos)
{
  NVAL num;
  if (!is_number(args[0])) {
    safe_str(T(e_num), buff, bp);
    return;
  }
  num = parse_number(args[0]);
  if ((num < -1) || (num > 1)) {
    safe_str(T(e_range), buff, bp);
    return;
  }
  safe_number(rad_to_angle(acos(num), args[1]), buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_tan)
{
  NVAL angle, t;
  if (!is_number(args[0])) {
    safe_str(T(e_num), buff, bp);
    return;
  }
  angle = angle_to_rad(parse_number(args[0]), args[1]);

  if (fmod(fabs(angle), M_PI_2) == 0.0 && fmod(fabs(angle), M_PI) != 0.0) {
    /* To infinity and beyond! */
    safe_str(T(e_range), buff, bp);
    return;
  }
#ifdef HAVE_FECLEAREXCEPT
  errno = 0;
  feclearexcept(FE_ALL_EXCEPT);
#endif

  t = tan(angle);

#ifdef HAVE_FETESTEXCEPT
  if (errno ||
      fetestexcept(FE_INVALID | FE_OVERFLOW | FE_UNDERFLOW | FE_DIVBYZERO)) {
    safe_str(T(e_range), buff, bp);
    return;
  }
#endif

  safe_number(t, buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_atan)
{
  NVAL angle;
  if (!is_number(args[0])) {
    safe_str(T(e_num), buff, bp);
    return;
  }
  angle = parse_number(args[0]);
  safe_number(rad_to_angle(atan(angle), args[1]), buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_atan2)
{
  NVAL x, y;
  if (!is_number(args[0]) || !is_number(args[1])) {
    safe_str(T(e_num), buff, bp);
    return;
  }
  x = parse_number(args[0]);
  y = parse_number(args[1]);
  safe_number(rad_to_angle(atan2(x, y), args[2]), buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_e)
{
  if (nargs == 0) {
    safe_number(2.71828182845904523536, buff, bp);
    return;
  }

  if (!is_number(args[0])) {
    safe_str(T(e_num), buff, bp);
    return;
  }
  safe_number(exp(parse_number(args[0])), buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_power)
{
  NVAL num, m, p;

  if (!is_number(args[0]) || !is_number(args[1])) {
    safe_str(T(e_nums), buff, bp);
    return;
  }
  num = parse_number(args[0]);
  m = parse_number(args[1]);

#ifdef HAVE_FECLEAREXCEPT
  errno = 0;
  feclearexcept(FE_ALL_EXCEPT);
#endif

  p = pow(num, m);

#ifdef HAVE_FETESTEXCEPT
  if (errno ||
      fetestexcept(FE_INVALID | FE_OVERFLOW | FE_UNDERFLOW | FE_DIVBYZERO)) {
    safe_str(T(e_range), buff, bp);
    return;
  }
#endif

  safe_number(p, buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_ln)
{
  NVAL num;
  if (!is_number(args[0])) {
    safe_str(T(e_num), buff, bp);
    return;
  }
  num = parse_number(args[0]);

  if (num < 0) {
    safe_str(T(e_range), buff, bp);
    return;
  }
  safe_number(log(num), buff, bp);
}

/* Cygwin has a log2 that configure doesn't detect. Macro, maybe? */
#if !defined(HAVE_LOG2) && !defined(__CYGWIN__)
static inline double
log2(double x)
{
  return log(x) / log(2.0);
}
#endif

/* ARGSUSED */
FUNCTION(fun_log)
{
  NVAL num;
  NVAL base = 10.0;
  bool base_is_e = false;

  if (!is_number(args[0])) {
    safe_str(T(e_nums), buff, bp);
    return;
  }
  num = parse_number(args[0]);

  if (num < 0) {
    safe_str(T(e_range), buff, bp);
    return;
  }
  if (nargs == 2) {
    if (!is_number(args[1])) {
      if (args[1][0] == 'e' && args[1][1] == '\0') {
        base_is_e = true;
      } else {
        safe_str(T(e_nums), buff, bp);
        return;
      }
    } else
      base = parse_number(args[1]);

    if (base_is_e)
      safe_number(log(num), buff, bp);
    else if (base <= 1)
      safe_str(T("#-1 BASE OUT OF RANGE"), buff, bp);
    else if (base == 10)
      safe_number(log10(num), buff, bp);
    else if (base == 2)
      safe_number(log2(num), buff, bp);
    else
      safe_number(log(num) / log(base), buff, bp);
  } else
    safe_number(log10(num), buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_sqrt)
{
  NVAL num;
  if (!is_number(args[0])) {
    safe_str(T(e_num), buff, bp);
    return;
  }
  num = parse_number(args[0]);
  if (num < 0) {
    safe_str(T("#-1 IMAGINARY NUMBER"), buff, bp);
    return;
  }
  safe_number(sqrt(num), buff, bp);
}

FUNCTION(fun_root)
{
  NVAL root, x;
  int n;
  int sign = 0;

  if (!is_number(args[0])) {
    safe_str(T(e_num), buff, bp);
    return;
  }
  if (!is_integer(args[1])) {
    safe_str(T(e_int), buff, bp);
    return;
  }
  x = parse_number(args[0]);
  n = parse_integer(args[1]);

  if (n < 0) {
    safe_str(T("#-1 ROOT OUT OF RANGE"), buff, bp);
    return;
  }

  if (x < 0) {
    if (n & 1) { /* Odd root */
      sign = 1;
      x = fabs(x);
    } else { /* Even */
      safe_str(T("#-1 IMAGINARY NUMBER"), buff, bp);
      return;
    }
  }

  switch (n) {
  case 0:
    root = 0;
    break;
  case 1:
    root = x;
    break;
  case 2:
    root = sqrt(x);
    break;
  case 3:
#ifdef HAVE_CBRT
    root = cbrt(x);
    break;
#endif
  default: {
    /* Root-finding algorithm detailed at
       http://en.wikipedia.org/wiki/Nth_root_algorithm */
    NVAL lastx, inverse_n, nm1, epsilon;
    int count = -1;
    /* See http://en.wikipedia.org/wiki/Talk:Nth_root_algorithm for
       initial guess equation, which is basically the old root-finding
       equation using 2 instead of e as a base. */
    root = pow(2.0, ceil(ceil(log2(x)) / n));
    inverse_n = 1.0 / n;
    nm1 = n - 1.0;
    epsilon = pow(0.1, FLOAT_PRECISION + 1);
    do {
      count += 1;
      lastx = root;
      root = inverse_n * ((nm1 * root) + (x / pow(root, nm1)));
    } while (fabs(lastx - root) > epsilon || count < 100);
  }
  }

  if (sign)
    root = -root;

  safe_number(root, buff, bp);
}

/** Calculates the numerator and denominator for a fraction representing
 *  a floating point number. Only works for positive numbers!
 * \param v the number
 * \param n pointer to the numerator
 * \param d pointer to the denominator
 * \param error accuracy to which the fraction should represent the original
 * number
 * \return -1.0 if (v < MIN || v > MAX || error < 0.0) | (v - n/d) / v |
 * otherwise.
 */
static double
frac(double v, double *RESTRICT n, double *RESTRICT d, double error)
{

  /* Based on a routine found in netlib (http://www.netlib.org) by

                          Robert J. Craig
                          AT&T Bell Laboratories
                          1200 East Naperville Road
                          Naperville, IL 60566-7045

   though I have no idea if that address is still valid.

   Rewritten by Raevnos to use modern C, and get rid of stupid gotos.

      reference:  Jerome Spanier and Keith B. Oldham, "An Atlas
      of Functions," Springer-Verlag, 1987, pp. 665-7.
    */

  double D, N, t;
  int first = 1;
  double epsilon, r = 0.0, m;

  if (v < 0 || error < 0.0)
    return -1.0;
  *d = D = 1;
  *n = floor(v);
  N = *n + 1;

  do {
    if (!first) {
      if (r <= 1.0)
        r = 1.0 / r;
      N += *n * floor(r);
      D += *d * floor(r);
      *n += N;
      *d += D;
    } else
      first = 0;
    r = 0.0;
    if (!EQ(v * (*d), *n)) {
      r = (N - v * D) / (v * (*d) - *n);
      if (r <= 1.0) {
        t = N;
        N = *n;
        *n = t;
        t = D;
        D = *d;
        *d = t;
      }
    }
    epsilon = fabs(1.0 - *n / (v * (*d)));
    if (epsilon <= error)
      return epsilon;
    m = 1.0;
    do {
      m *= 10.0;
    } while (m * epsilon < 1.0);
    epsilon = 1.0 / m * (floor(0.5 + m * epsilon));
    if (epsilon <= error)
      return epsilon;
  } while (!EQ(r, 0.0));
  return epsilon;
}

FUNCTION(fun_fraction)
{
  double num = 0, denom = 0;
  int whole = 0;
  NVAL n;

  if (!is_number(args[0])) {
    safe_str(T(e_num), buff, bp);
    return;
  }

  n = parse_number(args[0]);

  if (n < 0) {
    n = fabs(n);
    safe_chr('-', buff, bp);
  } else if (EQ(n, 0)) {
    safe_chr('0', buff, bp);
    return;
  }

  if (is_good_number(n)) {
    if (nargs > 1 && parse_boolean(args[1])) {
      whole = floor(n);
      n = n - whole;
    }
    frac(n, &num, &denom, 1.0e-10);

    if (fabs(denom - 1) < 1.0e-10) {
      if (whole)
        safe_integer(whole, buff, bp);
      else
        safe_format(buff, bp, "%.0f", num);
    } else {
      if (whole)
        safe_format(buff, bp, "%d %.0f/%.0f", whole, num, denom);
      else
        safe_format(buff, bp, "%.0f/%.0f", num, denom);
    }
  } else {
    safe_number(n, buff, bp);
  }
}

FUNCTION(fun_isint)
{
  bool valid = 1;
  char *end;
  if (arglens[0] == 0) {
    valid = 0;
  } else {
    errno = 0;
    parse_ival_full(args[0], &end, 10);
    if (errno || *end != '\0') {
      valid = 0;
    }
  }
  safe_boolean(valid, buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_isnum) { safe_boolean(is_strict_number(args[0]), buff, bp); }

/* ARGSUSED */
FUNCTION(fun_and) { math_and(args, nargs, buff, bp); }

/* ARGSUSED */
FUNCTION(fun_or) { math_or(args, nargs, buff, bp); }

/* ARGSUSED */
FUNCTION(fun_cand)
{
  int j;
  char tbuf[BUFFER_LEN], *tp;
  char const *sp;
  int negate = (called_as[0] == 'N');

  for (j = 0; j < nargs; j++) {
    tp = tbuf;
    sp = args[j];
    if (process_expression(tbuf, &tp, &sp, executor, caller, enactor, eflags,
                           PT_DEFAULT, pe_info))
      return;
    *tp = '\0';
    if (!parse_boolean(tbuf)) {
      safe_integer(negate, buff, bp);
      return;
    }
  }
  safe_integer(!negate, buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_cor)
{
  int j;
  char tbuf[BUFFER_LEN], *tp;
  char const *sp;
  int negate = (called_as[0] == 'N');

  for (j = 0; j < nargs; j++) {
    tp = tbuf;
    sp = args[j];
    if (process_expression(tbuf, &tp, &sp, executor, caller, enactor, eflags,
                           PT_DEFAULT, pe_info))
      return;
    *tp = '\0';
    if (parse_boolean(tbuf)) {
      safe_integer(!negate, buff, bp);
      return;
    }
  }
  safe_integer(negate, buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_not) { safe_boolean(!parse_boolean(args[0]), buff, bp); }

/* ARGSUSED */
FUNCTION(fun_t) { safe_boolean(parse_boolean(args[0]), buff, bp); }

/* ARGSUSED */
FUNCTION(fun_xor) { math_xor(args, nargs, buff, bp); }

/** Return the spelled-out version of an integer.
 * \param num string containing an integer to spell out.
 * \param len length of num, which must be a multiple of 3, max 15.
 * \param buff pointer to address of output buffer.
 * \param bp pointer to pointer to insertion point in buff.
 */
static void
do_spellnum(char *num, unsigned int len, char **buff, char ***bp)
{
  static const char *const bigones[] = {"", "thousand", "million", "billion",
                                        "trillion"};
  static const char *const singles[] = {
    "", "one", "two", "three", "four", "five", "six", "seven", "eight", "nine"};
  static const char *const special[] = {
    "ten",     "eleven",  "twelve",    "thirteen", "fourteen",
    "fifteen", "sixteen", "seventeen", "eighteen", "nineteen"};
  static const char *const tens[] = {"",       " ",     "twenty", "thirty",
                                     "forty",  "fifty", "sixty",  "seventy",
                                     "eighty", "ninety"};
  unsigned int x0, x1, x2;
  int pos = len / 3;
  int started = 0;

  while (pos > 0) {

    pos--;

    x0 = num[0] - '0';
    x1 = num[1] - '0';
    x2 = num[2] - '0';

    if (x0) {
      if (started)
        safe_chr(' ', *buff, *bp);
      safe_format(*buff, *bp, "%s %s", singles[x0], "hundred");
      started = 1;
    }

    if (x1 == 1) {
      if (started)
        safe_chr(' ', *buff, *bp);
      safe_str(special[x2], *buff, *bp);
      started = 1;
    } else if (x1 || x2) {
      if (started)
        safe_chr(' ', *buff, *bp);
      if (x1) {
        safe_str(tens[x1], *buff, *bp);
        if (x2)
          safe_chr('-', *buff, *bp);
      }
      safe_str(singles[x2], *buff, *bp);
      started = 1;
    }

    if (pos && (x0 || x1 || x2)) {
      if (started)
        safe_chr(' ', *buff, *bp);
      safe_str(bigones[pos], *buff, *bp);
      started = 1;
    }

    num += 3;
  }
} /* do_spellnum */

/** Convert the end of a spelled number string to ordinal form. */
static void
do_ordinalize(char **buff, char ***bp)
{
  char *p;
  size_t i, len;
  static const char *singles[] = {"one",  "two",    "three", "four",
                                  "five", "six",    "seven", "eight",
                                  "nine", "twelve", NULL};
  static const char *singleths[] = {"first", "second",  "third",   "fourth",
                                    "fifth", "sixth",   "seventh", "eighth",
                                    "ninth", "twelfth", NULL};
  /* Examine the end of the string */
  for (i = 0; singles[i]; i++) {
    len = strlen(singles[i]);
    p = **bp - len;
    if ((p >= *buff) && !strncasecmp(p, singles[i], len)) {
      **bp = p;
      safe_str(singleths[i], *buff, *bp);
      return; /* done */
    }
  }
  /* The string didn't end with a single. How about a y? */
  p = **bp - 1;
  if ((p >= *buff) && (*p == 'y')) {
    **bp = p;
    safe_str("ieth", *buff, *bp);
    return;
  }
  /* Ok, tack on th */
  safe_str("th", *buff, *bp);
}

/** adds zeros to the beginning of the string, until its length is
 * a multiple of 3.
 */
#define add_zeros(p)                                                           \
  m = strlen(p) % 3;                                                           \
  switch (m) {                                                                 \
  case 0:                                                                      \
    strcpy(num, p);                                                            \
    break;                                                                     \
  case 1:                                                                      \
    num[0] = '0';                                                              \
    num[1] = '0';                                                              \
    strcpy(num + 2, p);                                                        \
    break;                                                                     \
  case 2:                                                                      \
    num[0] = '0';                                                              \
    strcpy(num + 1, p);                                                        \
    break;                                                                     \
  }                                                                            \
  p = num;

/* ARGSUSED */
FUNCTION(fun_spellnum)
{
  static const char *tail[] = {
    "",           "tenth",          "hundredth",
    "thousandth", "ten-thousandth", "hundred-thousandth",
    "millionth",  "ten-millionth",  "hundred-millionth",
    "billionth",  "ten-billionth",  "hundred-billionth",
    "trillionth", "ten-trillionth", "hundred-trillionth"};

  char num[BUFFER_LEN];
  char *number, /* the whole number (without -/+ sign and leading zeros) */
    *pnumber, *pnum1,
    *pnum2 = NULL; /* part 1 and 2 of the number respectively */
  bool minus = 0;  /* is the number negative? */
  size_t len, m, dot = 0, len1, len2; /* length of part 1 and 2 respectively */
  bool ordinal_mode;

  ordinal_mode = (strcmp(called_as, "ORDINAL") == 0);
  pnumber = trim_space_sep(args[0], ' ');

  /* Is the number negative? */
  if (pnumber[0] == '-') {
    minus = 1;
    pnumber++;
  } else if (pnumber[0] == '+') {
    pnumber++;
  }

  /* remove leading zeros */
  while (*pnumber == '0')
    pnumber++;

  pnum1 = number = pnumber;

  /* Is it a number?
   * If so, devide the number in two parts: pnum1.pnum2
   */
  len = strlen(number);
  for (m = 0; m < len; m++) {

    if (*pnumber == '.') {
      if (ordinal_mode) {
        /* Only integers may be ordinalized */
        safe_str(T(e_int), buff, bp);
        return;
      }
      if (dot) {
        safe_str(T(e_num), buff, bp);
        return;
      }
      dot = 1;         /* allow only 1 dot in a number */
      *pnumber = '\0'; /* devide the string */
      pnum2 = pnumber + 1;
    } else if (!isdigit(*pnumber)) {
      safe_str(T(e_num), buff, bp);
      return;
    }
    pnumber++;
  }

  add_zeros(pnum1);

  len1 = strlen(pnum1);
  len2 = (pnum2 == NULL ? 0 : strlen(pnum2));

  /* Max number is 999,999,999,999,999.999,999,999,999 */
  if (len1 > 15 || len2 > 14) {
    safe_str(T(e_range), buff, bp);
    return;
  }

  /* before the . */

  /* zero is special */
  if (*pnum1 == '\0') {
    if (len2 == 0)
      safe_str("zero", buff, bp);
  } else {
    if (minus)
      safe_str("negative ", buff, bp);
    do_spellnum(pnum1, len1, &buff, &bp);
  }

  if (ordinal_mode) {
    /* In this case, we're done right here. */
    do_ordinalize(&buff, &bp);
    return;
  }

  if (len2 > 0) {
    /* after the . */

    /* remove leading zeros */
    while (*pnum2 == '0')
      pnum2++;

    add_zeros(pnum2);
    pnumber = num;

    len = strlen(pnumber);

    if (len1 > 0)
      safe_str(" and ", buff, bp);
    else if (minus && len)
      safe_str("negative ", buff, bp);

    /* zero is special */
    if (!len) {
      safe_str("zero ", buff, bp);
      safe_format(buff, bp, "%ss", tail[len2]);
    } else
      /* one is special too */
      if (len == 3 && parse_integer(pnumber) == 1) {
      safe_format(buff, bp, "one %s", tail[len2]);
    } else {
      /* rest is normal */
      do_spellnum(pnum1, len, &buff, &bp);
      safe_format(buff, bp, " %ss", tail[len2]);
    }
  }
}

#undef add_zeros

FUNCTION(fun_bound)
{
  if (!is_number(args[0]) || !is_number(args[1]) ||
      (nargs == 3 && !is_number(args[2]))) {
    safe_str(T(e_nums), buff, bp);
    return;
  }

  if (parse_number(args[0]) < parse_number(args[1]))
    safe_strl(args[1], arglens[1], buff, bp);
  else if (nargs == 3 && parse_number(args[0]) > parse_number(args[2]))
    safe_strl(args[2], arglens[2], buff, bp);
  else
    safe_strl(args[0], arglens[0], buff, bp);
}

FUNCTION(fun_band) { math_band(args, nargs, buff, bp); }

FUNCTION(fun_bnand)
{
  UIVAL retval;
  if (!is_uival(args[0]) || !is_uival(args[1])) {
    safe_str(T(e_uints), buff, bp);
    return;
  }
  retval = parse_uival(args[0]) & (~parse_uival(args[1]));
  safe_uinteger(retval, buff, bp);
}

FUNCTION(fun_bor) { math_bor(args, nargs, buff, bp); }

FUNCTION(fun_bxor) { math_bxor(args, nargs, buff, bp); }

FUNCTION(fun_bnot)
{
  if (!is_uival(args[0])) {
    safe_str(T(e_uint), buff, bp);
    return;
  }
  safe_uinteger(~parse_uival(args[0]), buff, bp);
}

/* ARGSUSED */
FUNCTION(fun_nand) { math_nand(args, nargs, buff, bp); }

/* ARGSUSED */
FUNCTION(fun_nor) { math_nor(args, nargs, buff, bp); }

/* ARGSUSED */
FUNCTION(fun_lmath)
{
  /* lmath(<op>, <list>[, <sep>])
   * is equivalant to
   *
   * &op me=<op>(%0, %1)
   * fold(me/op, <list>, <sep>)
   *
   * but a lot more efficient. The Tiny l-OP functions
   * can be simulated with @function if needed.
   */

  int nptr;
  char sep;
  char **ptr;
  const MATH *op;

  if (!delim_check(buff, bp, nargs, args, 3, &sep)) {
    return;
  }

  /* Allocate memory */
  ptr = mush_calloc(BUFFER_LEN, sizeof(char *), "string");

  nptr = list2arr(ptr, BUFFER_LEN, args[1], sep, 1);

  op = math_hash_lookup(args[0], arglens[0]);

  if (!op) {
    safe_str(T("#-1 UNKNOWN OPERATION"), buff, bp);
    mush_free(ptr, "string");
    return;
  }
  op->func(ptr, nptr, buff, bp);

  mush_free(ptr, "string");
}

/* Walker probably needs to convert from_base_XX arrays to a form
   suitable for putting in utils/gentables.c. Copy & paste is not it. ;)  */
static const signed char from_base_64[256] = {
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, 62, -1, 62, -1, 63, 52, 53, 54, 55, 56, 57, 58, 59, 60,
  61, -1, -1, -1, -1, -1, -1, -1, 0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  10,
  11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, -1, -1, -1, -1,
  63, -1, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42,
  43, 44, 45, 46, 47, 48, 49, 50, 51, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1};

static const signed char to_base_64[] =
  "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_";

static const signed char from_base_36[256] = {
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 0,  1,  2,  3,  4,  5,  6,  7,  8,
  9,  -1, -1, -1, -1, -1, -1, -1, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
  21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, -1, -1, -1, -1,
  -1, -1, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26,
  27, 28, 29, 30, 31, 32, 33, 34, 35, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1};

static const signed char to_base_36[] = "0123456789abcdefghijklmnopqrstuvwxyz";

FUNCTION(fun_baseconv)
{
  unsigned long n;
  int isnegative = 0;
  int m;
  unsigned int from, to;
  char *ptr;
  char numbuff[BUFFER_LEN], *nbp;

  /* Base 36 by default. */
  const signed char *frombase = from_base_36;
  const signed char *tobase = to_base_36;

  if (!(is_integer(args[1]) && is_integer(args[2]))) {
    safe_str(T(e_ints), buff, bp);
    return;
  }

  from = parse_uinteger(args[1]);
  to = parse_uinteger(args[2]);

  if (from < 2 || from > 64) {
    safe_str(T("#-1 FROM BASE OUT OF RANGE"), buff, bp);
    return;
  }

  if (from > 36) {
    frombase = from_base_64;
  }

  if (to < 2 || to > 64) {
    safe_str(T("#-1 TO BASE OUT OF RANGE"), buff, bp);
    return;
  }

  if (to > 36) {
    tobase = to_base_64;
  }

  /* Parse it. */
  ptr = trim_space_sep(args[0], ' ');
  n = 0;
  if (ptr) {
    /* Hyphen-minus are always treated as digits in base 63/base 64. */
    if (from < 63 && to < 63 && *ptr == '-') {
      isnegative = 1;
      ptr++;
    }
    while (*ptr) {
      n *= from;
      if (frombase[*ptr] >= 0 && frombase[*ptr] < (int) from) {
        n += frombase[*ptr];
        ptr++;
      } else {
        safe_str(T("#-1 MALFORMED NUMBER"), buff, bp);
        return;
      }
    }
  }

  if (isnegative) {
    safe_chr('-', buff, bp);
  }

  /* Handle the 0-case. (And quickly handle < to_base case, too!) */
  if (n < to) {
    safe_chr(tobase[n], buff, bp);
    return;
  }

  nbp = numbuff;

  /* This comes out backwards. */
  while (n > 0) {
    m = n % to;
    n = n / to;
    safe_chr(tobase[m], numbuff, &nbp);
  }

  /* Reverse back onto buff. */
  nbp--;
  while (nbp >= numbuff) {
    safe_chr(*nbp, buff, bp);
    nbp--;
  }
}

MATH_FUNC(math_add)
{
  NVAL result = 0;
  int n;

  for (n = 0; n < nptr; n++) {
    if (!is_number(ptr[n])) {
      safe_str(T(e_nums), buff, bp);
      return;
    }
    result += parse_number(ptr[n]);
  }

  safe_number(result, buff, bp);
}

MATH_FUNC(math_and)
{
  int n;

  if (nptr == 0) {
    safe_chr('0', buff, bp);
    return;
  }

  for (n = 0; n < nptr; n++) {
    if (!parse_boolean(ptr[n])) {
      safe_chr('0', buff, bp);
      return;
    }
  }

  safe_chr('1', buff, bp);
  return;
}

MATH_FUNC(math_sub)
{
  /* Subtraction */
  NVAL result;
  int n;

  if (nptr < 1) {
    safe_chr('0', buff, bp);
    return;
  }

  if (!is_number(ptr[0])) {
    safe_str(T(e_nums), buff, bp);
    return;
  }

  result = parse_number(ptr[0]);

  for (n = 1; n < nptr; n++) {
    if (!is_number(ptr[n])) {
      safe_str(T(e_nums), buff, bp);
      return;
    }
    result -= parse_number(ptr[n]);
  }
  safe_number(result, buff, bp);
}

MATH_FUNC(math_mul)
{
  NVAL result;
  int n;
  /* Multiplication */

  if (nptr < 1) {
    safe_chr('0', buff, bp);
    return;
  }

  if (!is_number(ptr[0])) {
    safe_str(T(e_nums), buff, bp);
    return;
  }
  result = parse_number(ptr[0]);

  for (n = 1; n < nptr; n++) {
    if (!is_number(ptr[n])) {
      safe_str(T(e_nums), buff, bp);
      return;
    }
    result *= parse_number(ptr[n]);
  }
  safe_number(result, buff, bp);
}

MATH_FUNC(math_min)
{
  NVAL result;
  int n;

  if (nptr < 1) {
    safe_chr('0', buff, bp);
    return;
  }

  if (!is_number(ptr[0])) {
    safe_str(T(e_nums), buff, bp);
    return;
  }
  result = parse_number(ptr[0]);

  for (n = 1; n < nptr; n++) {
    NVAL test;
    if (!is_number(ptr[n])) {
      safe_str(T(e_nums), buff, bp);
      return;
    }
    test = parse_number(ptr[n]);
    result = (result > test) ? test : result;
  }
  safe_number(result, buff, bp);
}

MATH_FUNC(math_max)
{
  NVAL result;
  int n;

  if (nptr < 1) {
    safe_chr('0', buff, bp);
    return;
  }

  if (!is_number(ptr[0])) {
    safe_str(T(e_nums), buff, bp);
    return;
  }
  result = parse_number(ptr[0]);

  for (n = 1; n < nptr; n++) {
    NVAL test;
    if (!is_number(ptr[n])) {
      safe_str(T(e_nums), buff, bp);
      return;
    }
    test = parse_number(ptr[n]);
    result = (result > test) ? result : test;
  }
  safe_number(result, buff, bp);
}

MATH_FUNC(math_mean)
{
  NVAL result = 0, count = 0;
  int n;

  if (nptr < 1) {
    safe_chr('0', buff, bp);
    return;
  }

  for (n = 0; n < nptr; n++) {
    if (!is_number(ptr[n])) {
      safe_str(T(e_nums), buff, bp);
      return;
    }
    result += parse_number(ptr[n]);
    count++;
  }

  safe_number(result / count, buff, bp);
}

MATH_FUNC(math_div)
{
  /* Division, truncating to match remainder */
  IVAL divresult;
  int n;

  if (nptr < 1) {
    safe_chr('0', buff, bp);
    return;
  }

  if (!is_ival(ptr[0])) {
    safe_str(T(e_ints), buff, bp);
    return;
  }
  divresult = parse_ival(ptr[0]);

  for (n = 1; n < nptr; n++) {
    IVAL temp;
    lldiv_t q;

    if (!is_ival(ptr[n])) {
      safe_str(T(e_ints), buff, bp);
      return;
    }
    temp = parse_ival(ptr[n]);

    if (temp == 0) {
      safe_str(T("#-1 DIVISION BY ZERO"), buff, bp);
      return;
    }

    if (divresult == INT64_MIN && temp == -1) {
      safe_str(T("#-1 DOMAIN ERROR"), buff, bp);
      return;
    }

    q = lldiv(divresult, temp);
    divresult = q.quot;
  }
  safe_integer(divresult, buff, bp);
}

MATH_FUNC(math_floordiv)
{
  /* Division taking the floor, to match modulo */
  IVAL divresult;
  int n;

  if (nptr < 1) {
    safe_chr('0', buff, bp);
    return;
  }

  if (!is_ival(ptr[0])) {
    safe_str(T(e_ints), buff, bp);
    return;
  }
  divresult = parse_ival(ptr[0]);

  for (n = 1; n < nptr; n++) {
    IVAL temp;
    if (!is_ival(ptr[n])) {
      safe_str(T(e_ints), buff, bp);
      return;
    }
    temp = parse_ival(ptr[n]);

    if (temp == 0) {
      safe_str(T("#-1 DIVISION BY ZERO"), buff, bp);
      return;
    }

    if (divresult == INT_MIN && temp == -1) {
      safe_str(T("#-1 DOMAIN ERROR"), buff, bp);
      return;
    }

    if (divresult < 0) {
      if (temp < 0)
        divresult = -divresult / -temp;
      else
        divresult = -((-divresult + temp - 1) / temp);
    } else {
      if (temp < 0)
        divresult = -((divresult - temp - 1) / -temp);
      else
        divresult = divresult / temp;
    }
  }
  safe_integer(divresult, buff, bp);
}

MATH_FUNC(math_fdiv)
{
  NVAL result;
  int n;
  /* Floating-point division */

  if (nptr < 1) {
    safe_chr('0', buff, bp);
    return;
  }

  if (!is_number(ptr[0])) {
    safe_str(T(e_nums), buff, bp);
    return;
  }
  result = parse_number(ptr[0]);

  for (n = 1; n < nptr; n++) {
    NVAL temp;
    if (!is_number(ptr[n])) {
      safe_str(T(e_nums), buff, bp);
      return;
    }
    temp = parse_number(ptr[n]);

    if (EQ(temp, 0)) {
      safe_str(T("#-1 DIVISION BY ZERO"), buff, bp);
      return;
    }

    result /= temp;
  }
  safe_number(result, buff, bp);
}

MATH_FUNC(math_modulo)
{
  /* Modulo */
  IVAL divresult;
  int n;

  if (nptr < 1) {
    safe_chr('0', buff, bp);
    return;
  }

  if (!is_ival(ptr[0])) {
    safe_str(T(e_ints), buff, bp);
    return;
  }
  divresult = parse_ival(ptr[0]);

  for (n = 1; n < nptr; n++) {
    IVAL temp;
    if (!is_ival(ptr[n])) {
      safe_str(T(e_ints), buff, bp);
      return;
    }
    temp = parse_ival(ptr[n]);

    if (temp == 0) {
      safe_str(T("#-1 DIVISION BY ZERO"), buff, bp);
      return;
    }

    if (divresult == INT_MIN && temp == -1) {
      safe_str(T("#-1 DOMAIN ERROR"), buff, bp);
      return;
    }

    if (divresult < 0) {
      if (temp < 0)
        divresult = -(-divresult % -temp);
      else
        divresult = (temp - (-divresult % temp)) % temp;
    } else {
      if (temp < 0)
        divresult = -((-temp - (divresult % -temp)) % -temp);
      else
        divresult = divresult % temp;
    }
  }
  safe_integer(divresult, buff, bp);
}

MATH_FUNC(math_remainder)
{
  /* Remainder */
  IVAL divresult;
  int n;

  if (nptr < 1) {
    safe_chr('0', buff, bp);
    return;
  }

  if (!is_ival(ptr[0])) {
    safe_str(T(e_ints), buff, bp);
    return;
  }
  divresult = parse_ival(ptr[0]);

  for (n = 1; n < nptr; n++) {
    IVAL temp;
    lldiv_t r;

    if (!is_ival(ptr[n])) {
      safe_str(T(e_ints), buff, bp);
      return;
    }
    temp = parse_ival(ptr[n]);

    if (temp == 0) {
      safe_str(T("#-1 DIVISION BY ZERO"), buff, bp);
      return;
    }

    if (divresult == INT_MIN && temp == -1) {
      safe_str(T("#-1 DOMAIN ERROR"), buff, bp);
      return;
    }

    r = lldiv(divresult, temp);
    divresult = r.rem;
  }
  safe_integer(divresult, buff, bp);
}

MATH_FUNC(math_band)
{
  UIVAL bretval;
  int n;

  if (nptr < 1) {
    safe_chr('0', buff, bp);
    return;
  }

  if (!is_uival(ptr[0])) {
    safe_str(T(e_uints), buff, bp);
    return;
  }

  bretval = parse_uival(ptr[0]);

  for (n = 1; n < nptr; n++) {
    if (!is_uival(ptr[n])) {
      safe_str(T(e_uints), buff, bp);
      return;
    }
    bretval &= parse_uival(ptr[n]);
  }
  safe_uinteger(bretval, buff, bp);
}

MATH_FUNC(math_bor)
{
  UIVAL bretval;
  int n;

  if (nptr < 1) {
    safe_chr('0', buff, bp);
    return;
  }

  if (!is_uival(ptr[0])) {
    safe_str(T(e_uints), buff, bp);
    return;
  }

  bretval = parse_uival(ptr[0]);

  for (n = 1; n < nptr; n++) {
    if (!is_uival(ptr[n])) {
      safe_str(T(e_uints), buff, bp);
      return;
    }
    bretval |= parse_uival(ptr[n]);
  }
  safe_uinteger(bretval, buff, bp);
}

MATH_FUNC(math_bxor)
{
  UIVAL bretval;
  int n;

  if (nptr < 1) {
    safe_chr('0', buff, bp);
    return;
  }

  if (!is_uival(ptr[0])) {
    safe_str(T(e_uints), buff, bp);
    return;
  }

  bretval = parse_uival(ptr[0]);

  for (n = 1; n < nptr; n++) {
    if (!is_uival(ptr[n])) {
      safe_str(T(e_uints), buff, bp);
      return;
    }
    bretval ^= parse_uival(ptr[n]);
  }
  safe_uinteger(bretval, buff, bp);
}

MATH_FUNC(math_or)
{
  int n;
  /* Or */

  for (n = 0; n < nptr; n++) {
    if (parse_boolean(ptr[n])) {
      safe_chr('1', buff, bp);
      return;
    }
  }
  safe_chr('0', buff, bp);
}

MATH_FUNC(math_nor)
{
  int n;
  /* nor */

  for (n = 0; n < nptr; n++) {
    if (parse_boolean(ptr[n])) {
      safe_chr('0', buff, bp);
      return;
    }
  }
  safe_chr('1', buff, bp);
}

MATH_FUNC(math_nand)
{
  int n;

  for (n = 0; n < nptr; n++) {
    if (!parse_boolean(ptr[n])) {
      safe_chr('1', buff, bp);
      return;
    }
  }
  safe_chr('0', buff, bp);
}

MATH_FUNC(math_xor)
{
  int found = 0, n;

  for (n = 0; n < nptr; n++) {
    if (parse_boolean(ptr[n])) {
      if (found == 0) {
        found = 1;
      } else {
        safe_chr('0', buff, bp);
        return;
      }
    }
  }
  if (found)
    safe_chr('1', buff, bp);
  else
    safe_chr('0', buff, bp);
}

MATH_FUNC(math_lt) { lmathcomp(ptr, nptr, buff, bp, 0, 0); }

MATH_FUNC(math_gt) { lmathcomp(ptr, nptr, buff, bp, 0, 1); }

MATH_FUNC(math_lte) { lmathcomp(ptr, nptr, buff, bp, 1, 0); }

MATH_FUNC(math_gte) { lmathcomp(ptr, nptr, buff, bp, 1, 1); }

MATH_FUNC(math_eq)
{
  /* Yes, I'm evil. :D  */
  lmathcomp(ptr, nptr, buff, bp, 1, -1);
}

/** This is used for lt, gt, lte, gte, eq */
static void
lmathcomp(char **ptr, int nptr, char *buff, char **bp, int eqokay, int isgt)
{
  NVAL prev = 0;
  NVAL next = 0;
  int n;

  if (nptr < 2) {
    safe_str(T("#-1 COMPARISON REQUIRES 2 OR MORE NUMBERS"), buff, bp);
    return;
  }

  if (!is_number(ptr[0])) {
    safe_str(T(e_nums), buff, bp);
    return;
  }
  prev = parse_number(ptr[0]);
  for (n = 1; n < nptr; n++, prev = next) {
    if (!is_number(ptr[n])) {
      safe_str(T(e_nums), buff, bp);
      return;
    }
    next = parse_number(ptr[n]);
    /* Is eqok? */
    if (EQ(next, prev)) {
      if (eqokay)
        continue;
      safe_chr('0', buff, bp);
      return;
    }
    if ((prev > next) != isgt) {
      safe_chr('0', buff, bp);
      return;
    }
  }

  safe_chr('1', buff, bp);
}

MATH_FUNC(math_neq)
{
  NVAL prev = 0;
  NVAL next = 0;
  int n;

  if (nptr < 2) {
    safe_str(T("#-1 COMPARISON REQUIRES 2 OR MORE NUMBERS"), buff, bp);
    return;
  }

  if (!is_number(ptr[1])) {
    safe_str(T(e_nums), buff, bp);
    return;
  }
  prev = parse_number(ptr[0]);
  for (n = 1; n < nptr; n++, prev = next) {
    if (!is_number(ptr[n])) {
      safe_str(T(e_nums), buff, bp);
      return;
    }
    next = parse_number(ptr[n]);
    if (!EQ(next, prev)) {
      safe_chr('1', buff, bp);
      return;
    }
  }

  safe_chr('0', buff, bp);
}

static NVAL
find_median(NVAL *numbers, int nargs)
{
  if (nargs == 0)
    return 0;
  if (nargs == 1)
    return numbers[0];

  qsort(numbers, nargs, sizeof(NVAL), nval_comp);

  if ((nargs % 2) == 1) /* Odd # of items */
    return numbers[nargs / 2];
  else
    return (numbers[(nargs / 2) - 1] + numbers[nargs / 2]) / (NVAL) 2;
}

FUNCTION(fun_median) { math_median(args, nargs, buff, bp); }

FUNCTION(fun_mean) { math_mean(args, nargs, buff, bp); }

FUNCTION(fun_stddev) { math_stddev(args, nargs, buff, bp); }

/* ARGSUSED */
MATH_FUNC(math_median)
{
  NVAL median;
  NVAL *numbers;
  int n;

  numbers = mush_malloc(nptr * sizeof(NVAL), "number_array");

  for (n = 0; n < nptr; n++) {
    if (!is_number(ptr[n])) {
      safe_str(T(e_nums), buff, bp);
      mush_free(numbers, "number_array");
      return;
    }
    numbers[n] = parse_number(ptr[n]);
  }

  median = find_median(numbers, nptr);
  mush_free(numbers, "number_array");
  safe_number(median, buff, bp);
}

MATH_FUNC(math_stddev)
{
  NVAL m, om, s, os, v;
  int n;

  if (nptr < 2) {
    safe_number(0, buff, bp);
    return;
  }
  if (!is_number(ptr[0])) {
    safe_str(T(e_nums), buff, bp);
    return;
  }
  m = parse_number(ptr[0]);
  s = 0;
  for (n = 1; n < nptr; n++) {
    om = m;
    os = s;
    if (!is_number(ptr[n])) {
      safe_str(T(e_nums), buff, bp);
      return;
    }
    v = parse_number(ptr[n]);
    m = om + (v - om) / (n + 1);
    s = os + (v - om) * (v - m);
  }

  safe_number(sqrt(s / (nptr - 1)), buff, bp);
}

MATH_FUNC(math_dist2d)
{
  NVAL d1, d2;

  if (nptr != 4) {
    safe_str(T("#-1 FUNCTION (DIST2D) EXPECTS 4 ARGUMENTS"), buff, bp);
    return;
  }

  if (!is_number(ptr[0]) || !is_number(ptr[1]) || !is_number(ptr[2]) ||
      !is_number(ptr[3])) {
    safe_str(T(e_nums), buff, bp);
    return;
  }
  d1 = parse_number(ptr[0]) - parse_number(ptr[2]);
  d2 = parse_number(ptr[1]) - parse_number(ptr[3]);
#ifdef HAVE_HYPOT
  safe_number(hypot(d1, d2), buff, bp);
#else
  safe_number(sqrt(d1 * d1 + d2 * d2), buff, bp);
#endif
}

MATH_FUNC(math_dist3d)
{
  NVAL d1, d2, d3, r;

  if (nptr != 6) {
    safe_str(T("#-1 FUNCTION (DIST3D) EXPECTS 6 ARGUMENTS"), buff, bp);
    return;
  }

  if (!is_number(ptr[0]) || !is_number(ptr[1]) || !is_number(ptr[2]) ||
      !is_number(ptr[3]) || !is_number(ptr[4]) || !is_number(ptr[5])) {
    safe_str(T(e_nums), buff, bp);
    return;
  }
  d1 = parse_number(ptr[0]) - parse_number(ptr[3]);
  d2 = parse_number(ptr[1]) - parse_number(ptr[4]);
  d3 = parse_number(ptr[2]) - parse_number(ptr[5]);
  r = d1 * d1 + d2 * d2 + d3 * d3;
  safe_number(sqrt(r), buff, bp);
}

static NVAL
angle_to_rad(NVAL angle, const char *from)
{
  if (!from)
    return angle;
  switch (*from) {
  case 'r':
  case 'R':
    return angle;
  case 'd':
  case 'D':
    return angle * (M_PI / 180.0);
  case 'g':
  case 'G':
    return angle * (M_PI / 200.0);
  default:
    return angle;
  }
}

static NVAL
rad_to_angle(NVAL angle, const char *to)
{
  if (!to)
    return angle;
  switch (*to) {
  case 'r':
  case 'R':
    return angle;
  case 'd':
  case 'D':
    return angle * (180.0 / M_PI);
  case 'g':
  case 'G':
    return angle * (200.0 / M_PI);
  default:
    return angle;
  }
}
